/** @file std.math.d
  * @brief Implements the vec2, vec3, vec4, mat2, mat3, and mat4 structs.
  * These more/less come from OpenGL 2.0 and have a bloody lot of functionality.
  */

/** @TODO Array operations.
  */

module net.BurtonRadons.dig.common.math;

private import net.BurtonRadons.dig.platform.base;

/** A 2d float point. */
struct vec2
{
    float x; /**< X coordinate. */
    float y; /**< Y coordinate. */
    
/+
#ifdef DoxygenMustSkipThis
+/

    bit digCommonBumpUpTheSize;
    
/+
#endif
+/
    
    /** A completely NAN vector. */
    const vec2 nan = { float.nan, float.nan };

/* Static functions. */

    /** Create a vector. */
    static vec2 create (float x, float y) { vec2 vector; vector.set (x, y); return vector; }

    /** Linear interpolation between two vectors.
      * @param a The vector to return when #t is 0.
      * @param b The vector to return when #t is 1.
      * @param t The amount to interpolate between #a and #b, from 0 (returns #a) to 1 (returns #b).  Out-of-range values are clamped.
      * @return The interpolated vector.
      */
    static vec2 lerp (vec2 a, vec2 b, float t)
    {
        if (t < 0) return a;
        if (t > 1) return b;
        return avec2 (a.x * (1 - t) + b.x * t, a.y * (1 - t) + b.y * t);
    }

/* Base functions. */

    /** Get this vector as an array. */
    float [] array () { return (&x) [0 .. 2]; }

    /** Index a parameter. */
    float index (int i) { return i == 0 ? x : y; }

    /** Set a parameter. */
    void index (int i, float v) { if (i == 0) x = v; else y = v; }

    /** Set all parameters to a std.math.single value. */
    void set (float v) { set (v, v); }

    /** Set the parameters. */
    void set (float x, float y) { this.x = x; this.y = y; }

    /** Swizzle */
    vec2 swizzle (int a, int b) { vec2 copy; copy.set (index (a), index (b)); return copy; }

    /** Return whether any parameter is nan. */
    bit isnan ()
    {
        return x != x || y != y;
    }

    /** Convert to std.string. */
    char [] toString ()
    {
        char [64] buffer;
        int length;

        length = std.c.stdio.sprintf (buffer, "vec2 (%g, %g)", x, y);
        return buffer [0 .. length].dup;
    }

/* Operations. */

    /** Get the absolute vector. */
    vec2 abs ()
    {
        return create (digCommonFloatAbs (x), digCommonFloatAbs (y));
    }

    /** Get the absolute maximum axis. */
    float absMaxAxis () { return digCommonFloatAbs (x) > digCommonFloatAbs (y) ? x : y; }

    /** Return the nearest integer that is equal to or greater than a. */
    vec2 ceil () { return create (std.math.ceil (x), std.math.ceil (y)); }

    /** Return min (max (a, minVal), maxVal). */
    vec2 clamp (vec2 minVal, vec2 maxVal) { return create (fmid (minVal.x, x, maxVal.x), fmid (minVal.y, y, maxVal.y)); }

    /** Return min (max (a, minVal), maxVal). */
    vec2 clamp (float minVal, float maxVal) { return create (fmid (minVal, x, maxVal), fmid (minVal, y, maxVal)); }

    /** Return whether this vector is greater than or equal to b in each axis. */
    bit cmpge (vec2 b)
    {
        return x >= b.x && y >= b.y;
    }

    /** Return whether this vector is greater than b in each axis. */
    bit cmpg (vec2 b)
    {
        return x > b.x && y > b.y;
    }

    /** Return whether this vector is less than b in each axis. */
    bit cmpl (vec2 b)
    {
        return x < b.x && y < b.y;
    }

    /** Return whether this vector is less than or equal to b in each axis. */
    bit cmple (vec2 b)
    {
        return x <= b.x && y <= b.y;
    }

    /** Get the distance to the other vector. */
    float distance (vec2 b) { return std.math.sqrt (squaredDistance (b)); }

    /** Return the nearest integer that is equal to or less than a. */
    vec2 floor () { return create (std.math.floor (x), std.math.floor (y)); }

    /** Return a - a.floor (). */
    vec2 fract () { return create (x - std.math.floor (x), y - std.math.floor (y)); }

    /** Get the magnitude. */
    float magnitude () { return std.math.sqrt (squaredMagnitude ()); }

    /** Select the maximum of either's axis. */
    vec2 max (vec2 b) { return create (fmax (x, b.x), fmax (y, b.y)); }

    /** Get the maximum axis. */
    float maxAxis () { return fmax (x, y); }

    /** Select the minimum of either's axis. */
    vec2 min (vec2 b) { return create (fmin (x, b.x), fmin (y, b.y)); }

    /** Return the value of the minimum signed axis. */
    float minAxis () { return fmin (x, y); }

    /** Return the linear blend of this and b (this * (1 - t) + b * t). */
    vec2 mix (vec2 b, float t)
    {
        vec2 ma = opMul (1 - t);
        vec2 mb = b * t;

        return ma + mb;
    }

    /** Return the unit vector. */
    vec2 normalize ()
    {
        return *this / magnitude ();
    }

    /** Return 1.0 if a >= 0, else -1.0. */
    vec2 sign () { return create (x >= 0 ? 1.0 : -1.0, y >= 0 ? 1.0 : -1.0); }

    /** Get the distance to the other vector, squared. */
    float squaredDistance (vec2 b) { return (x - b.x) * (x - b.x) + (y - b.y) * (y - b.y); }

    /** Get the squared magnitude. */
    float squaredMagnitude () { return x * x + y * y; }

/** @name Operator overloading.
  *//**@{*/

    /** Add vectors. */
    vec2 opAdd (vec2 b)
    {
        return create (x + b.x, y + b.y);
    }

    /** Add scalar. */
    vec2 opAdd (float b) { return create (x + b, y + b); }

    /** Add vector and assign. */
    vec2 opAddAssign (vec2 b) { x += b.x; y += b.y; return *this; }

    /** Add scalar and assign. */
    vec2 opAddAssign (float b) { x += b; y += b; return *this; }

    /** Subtract vectors. */
    vec2 opSub (vec2 b) { return create (x - b.x, y - b.y); }

    /** Subtract scalar. */
    vec2 opSub (float b) { return create (x - b, y - b); }

    /** Subtract vector and assign. */
    vec2 opSubAssign (vec2 b) { x -= b.x; y -= b.y; return *this; }

    /** Subtract scalar and assign. */
    vec2 opSubAssign (float b) { x -= b; y -= b; return *this; }

    /** Multiply vector. */
    vec2 opMul (vec2 b) { return create (x * b.x, y * b.y); }

    /** Multiply scalar. */
    vec2 opMul (float b) { return create (x * b, y * b); }

    /** Multiply vector and assign. */
    vec2 opMulAssign (vec2 b) { x *= b.x; y *= b.y; return *this; }

    /** Multiply scalar and assign. */
    vec2 opMulAssign (float b) { x *= b; y *= b; return *this; }

    /** Divide vector. */
    vec2 opDiv (vec2 b) { return create (x / b.x, y / b.y); }

    /** Divide scalar. */
    vec2 opDiv (float b) { return create (x / b, y / b); }

    /** Divide vector and assign. */
    vec2 opDivAssign (vec2 b) { x /= b.x; y /= b.y; return *this; }

    /** Divide scalar and assign. */
    vec2 opDivAssign (float b) { x /= b; y /= b; return *this; }

    /** Negate each axis. */
    vec2 opNeg () { return create (-x, -y); }

    /** Return equality with a vector. */
    bit opEqual (vec2 b) { return x == b.x && y == b.y; }
    
    /** Return equality with a scalar. */
    bit opEqual (float v) { return x == v && y == v; }

    /** Compare with a vector. */
    float opCmp (vec2 b)
    {
        if (x < b.x && y < b.y)
            return -1;
        if (x > b.x && y > b.y)
            return 1;
        if (x == b.x && y == b.y)
            return 0;
        return float.nan;
    }

/**@}*/
}

/** Create a two-dimensional vector. 
  * @relates vec2
  */
vec2 avec2 (float x, float y)
{
    return vec2.create (x, y);
}

/** Create a two-dimensional vector by stripping off z.
  * @relates vec2
  */
vec2 avec2 (vec3 vector)
{
    return vec2.create (vector.x, vector.y);
}

/** Create a two-dimensional vector by stripping off z and w.
  * @relates vec2
  */
vec2 avec2 (vec4 vector)
{
    return vec2.create (vector.x, vector.y);
}

/** A 3d float point. */
struct vec3
{
    float x; /**< X coordinate. */
    float y; /**< Y coordinate. */
    float z; /**< Z coordinate. */

    /** A completely NAN vector. */
    const vec3 nan = { float.nan, float.nan, float.nan };

/* Static functions. */

    /** Create a vector. */
    static vec3 create (float x, float y) { vec3 vector; vector.set (x, y); return vector; }

    /** Create a vector. */
    static vec3 create (float x, float y, float z) { vec3 vector; vector.set (x, y, z); return vector; }

    /** Linear interpolation between two vectors.
      * @param a The vector to return when #t is 0.
      * @param b The vector to return when #t is 1.
      * @param t The amount to interpolate between #a and #b, from 0 (returns #a) to 1 (returns #b).  Out-of-range values are clamped.
      * @return The interpolated vector.
      */
    static vec3 lerp (vec3 a, vec3 b, float t)
    {
        if (t < 0) return a;
        if (t > 1) return b;
        return avec3 (a.x * (1 - t) + b.x * t, a.y * (1 - t) + b.y * t, a.z * (1 - t) + b.z * t);
    }

/** @name Basic operations.
  *//**@{*/

    /** Get this vector as an array. */
    float [] array () { return (&x) [0 .. 3]; }

    /** Index a parameter. */
    float index (int i) { return i == 0 ? x : i == 1 ? y : z; }

    /** Set a parameter. */
    void index (int i, float v) { if (i == 0) x = v; else if (i == 1) y = v; else z = v; }

    /** Return whether any parameter is nan. */
    bit isnan ()
    {
        return x != x || y != y || z != z;
    }

    /** Set all parameters to a std.math.single value. */
    void set (float v) { set (v, v, v); }

    /** Set the parameters and z to zero. */
    void set (float x, float y) { this.x = x; this.y = y; this.z = 0; }

    /** Set the parameters. */
    void set (float x, float y, float z) { this.x = x; this.y = y; this.z = z; }

    /** Swizzle and return a 2d vector. */
    vec2 swizzle (int a, int b) { vec2 copy; copy.set (index (a), index (b)); return copy; }

    /** Swizzle and return a 3d vector. */
    vec3 swizzle (int a, int b, int c) { vec3 copy; copy.set (index (a), index (b), index (c)); return copy; }

///@}
/** @name Operations on vectors.
  *//**@{*/

    /** Get the absolute vector. */
    vec3 abs () { return create (digCommonFloatAbs (x), digCommonFloatAbs (y), digCommonFloatAbs (z)); }

    /** Get the absolute maximum axis. */
    float absMaxAxis ()
    {
        vec3 a = abs ();
        return a.x > a.y ? (a.x > a.z ? x : z) : (a.y > a.z ? y : z);
    }

    /** Get the angular difference between these vectors in degrees. */
    float angularDifferenceDeg (vec3 b)
    {
        return angularDifferenceRad (b) * 180 / std.math.PI;
    }

    /** Get the angular difference between these vectors in radians. */
    float angularDifferenceRad (vec3 b)
    {
        double n, d;

        n = x * b.x + y * b.y + z * b.z;
        d = std.math.sqrt (x * x + y * y + z * z)
          * std.math.sqrt (b.x * b.x + b.y * b.y + b.z * b.z);
        return std.math.acos (n / d);
    }

    /** Return the nearest integer that is equal to or greater than a. */
    vec3 ceil () { return create (std.math.ceil (x), std.math.ceil (y), std.math.ceil (z)); }

    /** Return min (max (a, minVal), maxVal). */
    vec3 clamp (vec3 minVal, vec3 maxVal) { return create (fmid (minVal.x, x, maxVal.x), fmid (minVal.y, y, maxVal.y), fmid (minVal.z, z, maxVal.z)); }

    /** Return min (max (a, minVal), maxVal). */
    vec3 clamp (float minVal, float maxVal) { return create (fmid (minVal, x, maxVal), fmid (minVal, y, maxVal), fmid (minVal, z, maxVal)); }

    /** Return whether this vector is greater than or equal to b in each axis. */
    bit cmpge (vec3 b)
    {
        return x >= b.x && y >= b.y && z >= b.z;
    }

    /** Return whether this vector is greater than b in each axis. */
    bit cmpg (vec3 b)
    {
        return x > b.x && y > b.y && z > b.z;
    }

    /** Return whether this vector is less than b in each axis. */
    bit cmpl (vec3 b)
    {
        return x < b.x && y < b.y && z < b.z;
    }

    /** Return whether this vector is less than or equal to b in each axis. */
    bit cmple (vec3 b)
    {
        return x <= b.x && y <= b.y && z <= b.z;
    }

    /** Return whether these three points define a straight line within a certain tolerance. */
    static bit colinear (vec3 a, vec3 b, vec3 c, float tolerance)
    {
        if (std.math.fabs ((b.y - a.y) * (c.x - a.x) - (c.y - a.y) * (b.x - a.x)) < tolerance
         && std.math.fabs ((b.z - a.z) * (c.x - a.x) - (c.z - a.z) * (b.x - a.x)) < tolerance
         && std.math.fabs ((b.z - a.z) * (c.y - a.y) - (c.z - a.z) * (b.y - a.y)) < tolerance)
            return true;
        return false;
    }

    /** Return whether these three points define a straight line within a tolerance of 0.02. */
    static bit colinear (vec3 a, vec3 b, vec3 c)
    {
        return colinear (a, b, c, 0.02);
    }

    /** Get the cross product. */
    vec3 cross (vec3 b)
    {
        return create (y * b.z - z * b.y,
                       z * b.x - x * b.z,
                       x * b.y - y * b.x);
    }

    /** Distance to the other vector. */
    float distance (vec3 o)
    {
        return std.math.sqrt ((x - o.x) * (x - o.x) + (y - o.y) * (y - o.y) + (z - o.z) * (z - o.z));
    }

    /** Get the dot product. */
    float dot (vec3 o)
    {
        return x * o.x + y * o.y + z * o.z;
    }

    /** Return the nearest integer that is equal to or less than a. */
    vec3 floor () { return create (std.math.floor (x), std.math.floor (y), std.math.floor (z)); }

    /** Return a - a.floor (). */
    vec3 fract () { return create (x - std.math.floor (x), y - std.math.floor (y), z - std.math.floor (z)); }

    /** Get the magnitude. */
    float magnitude () { return std.math.sqrt (squaredMagnitude ()); }

    /** Select the maximum of either's axis. */
    vec3 max (vec3 b) { return create (fmax (x, b.x), fmax (y, b.y), fmax (z, b.z)); }

    /** Get the maximum axis. */
    float maxAxis () { return fmax (x, y, z); }

    /** Select the minimum of either's axis. */
    vec3 min (vec3 b) { return create (fmin (x, b.x), fmin (y, b.y), fmin (z, b.z)); }

    /** Return the value of the minimum signed axis. */
    float minAxis () { return fmin (x, y, z); }

    /** Return the index of the minimum absolute axis. */
    int minAbsAxisIndex ()
    {
        if (std.math.fabs (x) < std.math.fabs (y))
        {
            if (std.math.fabs (x) < std.math.fabs (z))
                return 0;
            return 2;
        }
        if (std.math.fabs (y) < std.math.fabs (z))
            return 1;
        return 2;
    }

    /** Return the linear blend of this and b (this * (1 - t) + b * t). */
    vec3 mix (vec3 b, float t)
    {
        vec3 ma = opMul (1 - t);
        vec3 mb = b * t;

        return ma + mb;
    }

    /** Get the unit vector. */
    vec3 normalize ()
    {
        float m = 1.0 / magnitude ();
    
        return create (x * m, y * m, z * m);
    }

    /** Refract the vector.  This vector is the surface point.
      * eye is the location of the eye; normal is the normal of this surface
      * (must be normalized).  refractIndex is the degree and angle of refraction.
      */

    vec3 refract (vec3 eye, vec3 normal, float refractIndex)
    {
        vec3 env = eye - *this;
        float ndote = normal.dot (env) / refractIndex;
        float ndotn = normal.dot (normal);
        vec3 refracted;

        refracted = normal * ndote;
        refracted -= env * ndotn;

        return refracted;
    }

    /** Return 1.0 if a >= 0, else -1.0. */
    vec3 sign ()
    {
        return create (x >= 0 ? 1.0 : -1.0, y >= 0 ? 1.0 : -1.0, z >= 0 ? 1.0 : -1.0);
    }

    /** Squared distance to the other vector. */
    float squaredDistance (vec3 o)
    {
        return (x - o.x) * (x - o.x) + (y - o.y) * (y - o.y) + (z - o.z) * (z - o.z);
    }

    /** Get the squared magnitude. */
    float squaredMagnitude () { return x * x + y * y + z * z; }

    /** Print out. */
    void print ()
    {
        printf ("vec3 (%g, %g, %g)", x, y, z);
    }

    /** Convert to std.string. */
    char [] toString ()
    {
        char [256] buffer;
        int length;

        length = std.c.stdio.sprintf (buffer, "vec3 (%g, %g, %g)", x, y, z);
        return buffer [0 .. length].dup;
    }

/**@}*/
/** @name Operator overloading.
  *//**@{*/

    /** Add vector. */
    vec3 opAdd (vec3 b) { return create (x + b.x, y + b.y, z + b.z); }

    /** Add zero-extended vector. */
    vec3 opAdd (vec2 b) { return create (x + b.x, y + b.y, z); }

    /** Add scalar. */
    vec3 opAdd (float v) { return create (x + v, y + v, z + v); }

    /** Add vector and assign. */
    vec3 opAddAssign (vec3 b) { x += b.x; y += b.y; z += b.z; return *this; }

    /** Add zero-extended vector. */
    vec3 opAddAssign (vec2 b) { x += b.x; y += b.y; return *this; }

    /** Add scalar and assign. */
    vec3 opAddAssign (float v) { x += v; y += v; z += v; return *this; }
    
    /** Compare zero-extended vector. */
    int opCmp (vec2 b)
    {
        if (x < b.x && y < b.y && z < 0)
            return -1;
        if (x > b.x && y > b.y && z > 0)
            return 1;
        if (x == b.x && y == b.y && z == 0)
            return 0;
        return float.nan;
    }

    /** Compare vectors. */
    float opCmp (vec3 b)
    {
        if (x < b.x && y < b.y && z < b.z)
            return -1;
        if (x > b.x && y > b.y && z > b.z)
            return 1;
        if (x == b.x && y == b.y && z == b.z)
            return 0;
        return float.nan;
    }

    /** Divide vector. */
    vec3 opDiv (vec3 b) { return create (x / b.x, y / b.y, z / b.z); }

    /** Divide zero-extended vector. */
    vec3 opDiv (vec2 b) { return create (x / b.x, y / b.y, float.nan); }

    /** Divide scalar. */
    vec3 opDiv (float value) { return create (x / value, y / value, z / value); }

    /** Divide vector and assign. */
    vec3 opDivAssign (vec3 b) { x /= b.x; y /= b.y; z /= b.z; return *this; }

    /** Divide zero-extended vector and assign. */
    vec3 opDivAssign (vec2 b) { x /= b.x; y /= b.y; z = float.nan; return *this; }

    /** Divide scalar and assign. */
    vec3 opDivAssign (float v) { x /= v; y /= v; z /= v; return *this; }

    /** Divide scalar by vector. */
    vec3 opDiv_r (float value) { return create (value / x, value / y, value / z); }

    /** Return equality with scalar. */
     bit opEqual (float v) { return x == v && y == v && z == v; }

    /** Return equality with zero-extended vector. */
     bit opEqual (vec2 b) { return x == b.x && y == b.y && z == 0; }

    /** Return equality with vector. */
     bit opEqual (vec3 b) { return x == b.x && y == b.y && z == b.z; }

    /** Modulus vector. */
    vec3  opMod (vec3 b) { return create (x % b.x, y % b.y, z % b.z); }

    /** Modulus zero-extended vector. */
    vec3  opMod (vec2 b) { return create (x % b.x, y % b.y, float.nan); }

    /** Modulus scalar. */
    vec3  opMod (float v) { return create (x % v, y % v, z % v); }

    /** Modulus vector and assign. */
    vec3 opModAssign (vec3 b) { x = x % b.x; y = y % b.y; z = z % b.z; return *this; }

    /** Modulus zero-extended vector and assign. */
    vec3 opModAssign (vec2 b) { x = x % b.x; y = y % b.y; z = float.nan; return *this; }

    /** Divide scalar and assign. */
    vec3 opModAssign (float v) { x = x % v; y = y % v; z = z % v; return *this; }

    /** Multiply vector. */
    vec3 opMul (vec3 b) { return create (x * b.x, y * b.y, z * b.z); }

    /** Multiply zero-extended vector. */
    vec3 opMul (vec2 b) { return create (x * b.x, y * b.y, 0); }

    /** Multiply scalar. */
    vec3 opMul (float v) { return create (x * v, y * v, z * v); }

    /** Multiply vector and assign. */
    vec3 opMulAssign (vec3 b) { x *= b.x; y *= b.y; z *= b.z; return *this; }

    /** Multiply zero-extended vector and assign. */
    vec3 opMulAssign (vec2 b) { x *= b.x; y *= b.y; z = 0; return *this; }

    /** Multiply scalar and assign. */
    vec3 opMulAssign (float v) { x *= v; y *= v; z *= v; return *this; }

    /** Negate each axis. */
    vec3 opNeg () { return create (-x, -y, -z); }

    /** Subtract vector. */
    vec3 opSub (vec3 b) { return create (x - b.x, y - b.y, z - b.z); }

    /** Subtract zero-extended vector. */
    vec3 opSub (vec2 b) { return create (x - b.x, y - b.y, z); }

    /** Subtract scalar. */
    vec3 opSub (float v) { return create (x - v, y - v, z - v); }

    /** Subtract vector and assign. */
    vec3 opSubAssign (vec3 b) { x -= b.x; y -= b.y; z -= b.z; return *this; }

    /** Subtract zero-extended vector. */
    vec3 opSubAssign (vec2 b) { x -= b.x; y -= b.y; return *this; }

    /** Subtract scalar and assign. */
    vec3 opSubAssign (float v) { x -= v; y -= v; z -= v; return *this; }

/**@}*/
}

/+
#ifdef DOXYGOHOME
br_extern (void) br_vector3d_cartesian_to_polar_deg (br_vector3d *self, const br_vector3d *vector); /**< Convert cartesian vector to polar degrees. */
br_extern (void) br_vector3d_cartesian_to_polar_rad (br_vector3d *self, const br_vector3d *vector); /**< Convert cartesian vector to polar radians. */
br_inline (int) br_vector3d_colinear (const br_vector3d *a, const br_vector3d *b, const br_vector3d *c);
br_extern (int) br_vector3d_in_polygon (const br_vector3d *vector, int count, const br_vector3d *points); /**< Determine whether point is in polygon */
br_inline (void) br_vector3d_lerp (br_vector3d *out, const br_vector3d *a, const br_vector3d *b, br_real t); /**< Linearly interpolate between a (t = 0) and b (t = 1). */
br_extern (void) br_vector3d_nearest_on_line (br_vector3d *vector, const br_vector3d *point, const br_vector3d *start, const br_vector3d *end); /**< Find the nearest point on a line segment */
br_extern (void) br_vector3d_polar_deg_to_cartesian (br_vector3d *self, const br_vector3d *vector); /**< Convert polar degrees vector to cartesian. */
br_extern (void) br_vector3d_polar_rad_to_cartesian (br_vector3d *self, const br_vector3d *vector); /**< Convert polar radian vector to cartesian. */

/** Determine whether two triangles overlap.
 * @param a First triangle to test, three vectors.
 * @param b Second triangle to test, three vectors.
 * @return Returns zero without overlap, one on overlap.
 */
 
br_extern (int)
br_vector3d_tri_tri_overlap (const br_vector3d *a, const br_vector3d *b);
#endif
+/

/** Create a three-dimensional vector. 
  * @relates vec3
  */
vec3 avec3 (float x, float y)
{
    return vec3.create (x, y);
}

/** Create a three-dimensional vector.
  * @relates vec3
  */
vec3 avec3 (float x, float y, float z)
{
    return vec3.create (x, y, z);
}

/** Expand a two-dimensional vector with a z of zero.
  * @relates vec3
  */
vec3 avec3 (vec2 vector)
{
    return vec3.create (vector.x, vector.y);
}

/** Create a three-dimensional vector by removing the w axis.
  * @relates vec3
  */
vec3 avec3 (vec4 vector)
{
    return vec3.create (vector.x, vector.y, vector.z);
}

/** A 4d float point. */
struct vec4
{
    float x; /**< X coordinate. */
    float y; /**< Y coordinate. */
    float z; /**< Z coordinate. */
    float w; /**< W coordinate. */

/* Static functions. */

    /** Create a vector with a z and w of zero. */
    static vec4 create (float x, float y) { vec4 vector; vector.set (x, y); return vector; }

    /** Create a vector with a w of zero. */
    static vec4 create (float x, float y, float z) { vec4 vector; vector.set (x, y, z); return vector; }

    /** Create a vector. */
    static vec4 create (float x, float y, float z, float w) { vec4 vector; vector.set (x, y, z, w); return vector; }

/** @name Base operations.
  *//**@{*/

    /** Get this vector as an array. */
    float [] array () { return (&x) [0 .. 4]; }

    /** Index a parameter. */
    float index (int i) { return i == 0 ? x : i == 1 ? y : i == 2 ? z : w; }

    /** Set an index. */
    void index (int i, float v) { if (i == 0) x = v; else if (i == 1) y = v; else if (i == 2) z = v; else w = v; }

    /** Print out. */
    void print ()
    {
        printf ("vec4 (%g, %g, %g, %g)\n", x, y, z, w);
    }

    /** Set all parameters to a std.math.single value. */
    void set (float v) { set (v, v, v, v); }

    /** Set the parameters and z and w to zero. */
    void set (float x, float y) { this.x = x; this.y = y; this.z = 0; this.w = 0; }

    /** Set the parameters and w to zero. */
    void set (float x, float y, float z) { this.x = x; this.y = y; this.z = z; this.w = 0; }

    /** Set the parameters. */
    void set (float x, float y, float z, float w) { this.x = x; this.y = y; this.z = z; this.w = w; }

    /** Swizzle */
    vec2 swizzle (int a, int b) { vec2 copy; copy.set (index (a), index (b)); return copy; }
    vec3 swizzle (int a, int b, int c) { vec3 copy; copy.set (index (a), index (b), index (c)); return copy; }
    vec4 swizzle (int a, int b, int c, int d) { vec4 copy; copy.set (index (a), index (b), index (c), index (d)); return copy; }

    /** Convert to std.string. */
    char [] toString ()
    {
        char [256] buffer;
        int length;

        length = std.c.stdio.sprintf (buffer, "vec4 (%g, %g, %g, %g)", x, y, z, w);
        return buffer [0 .. length].dup;
    }

/**@}*/

/** @name Operations on vectors.
  *//**@{*/

    /** Get the absolute vector. */
    vec4 abs () { return create (x < 0 ? -x : x, y < 0 ? -y : y, z < 0 ? -z : z, w < 0 ? -w : w); }

    /** Get the absolute maximum axis. */
    float absMaxAxis ()
    {
        vec4 a = abs ();
        return a.x > a.y ? (a.x > a.z ? (a.x > a.w ? x : w) : (a.z > a.w ? z : w)) : (a.y > a.z ? (a.y > a.w ? y : w) : (a.z > a.w ? z : w));
    }

    /** Return the nearest integer that is equal to or greater than a. */
    vec4 ceil () { return create (std.math.ceil (x), std.math.ceil (y), std.math.ceil (z), std.math.ceil (w)); }

    /** Return min (max (a, minVal), maxVal). */
    vec4 clamp (vec4 minVal, vec4 maxVal) { return create (fmid (minVal.x, x, maxVal.x), fmid (minVal.y, y, maxVal.y), fmid (minVal.z, z, maxVal.z), fmid (minVal.w, w, maxVal.w)); }

    /** Return min (max (a, minVal), maxVal). */
    vec4 clamp (float minVal, float maxVal) { return create (fmid (minVal, x, maxVal), fmid (minVal, y, maxVal), fmid (minVal, z, maxVal), fmid (minVal, w, maxVal)); }

    /** Return whether this vector is greater than or equal to b in each axis. */
    bit cmpge (vec3 b)
    {
        return x >= b.x && y >= b.y && z >= b.z;
    }

    /** Return whether this vector is greater than b in each axis. */
    bit cmpg (vec3 b)
    {
        return x > b.x && y > b.y && z > b.z;
    }

    /** Return whether this vector is less than b in each axis. */
    bit cmpl (vec3 b)
    {
        return x < b.x && y < b.y && z < b.z;
    }

    /** Return whether this vector is less than or equal to b in each axis. */
    bit cmple (vec3 b)
    {
        return x <= b.x && y <= b.y && z <= b.z;
    }

    /** Return the nearest integer that is equal to or less than a. */
    vec4 floor () { return create (std.math.floor (x), std.math.floor (y), std.math.floor (z), std.math.floor (w)); }

    /** Return a - a.floor (). */
    vec4 fract () { return create (x - std.math.floor (x), y - std.math.floor (y), z - std.math.floor (z), w - std.math.floor (w)); }

    /** Select the maximum of either's axis. */
    vec4 max (vec4 b) { return create (fmax (x, b.x), fmax (y, b.y), fmax (z, b.z), fmax (w, b.w)); }

    /** Get the maximum axis. */
    float maxAxis () { return fmax (x, y, z, w); }

    /** Select the minimum of either's axis. */
    vec4 min (vec4 b) { return create (fmin (x, b.x), fmin (y, b.y), fmin (z, b.z), fmin (w, b.w)); }

    /* Get the minimum axis. */
    float minAxis () { return fmin (x, y, z, w); }

    /** Return the linear blend of this and b (this * (1 - t) + b * t). */
    vec4 mix (vec4 b, float t)
    {
        vec4 ma = opMul (1 - t);
        vec4 mb = b * t;

        return ma + mb;
    }

    /** Return 1.0 if a >= 0, else -1.0. */
    vec4 sign ()
    {
        return create (x >= 0 ? 1.0 : -1.0, y >= 0 ? 1.0 : -1.0, z >= 0 ? 1.0 : -1.0, w >= 0 ? 1.0 : -1.0);
    }

/**@}*/
/** @name Operator overloading.
  *//**@{*/

    vec4 opMul (vec4 b) { return create (x * b.x, y * b.y, z * b.z, w * b.w); }

    vec4 opMul (float v) { return create (x * v, y * v, z * v, w * v); }

/**@}*/
}

/** Create a four-dimensional vector with zero in z and w.
  * @relates vec4
  */
vec4 avec4 (float x, float y)
{
    return vec4.create (x, y);
}

/** Create a four-dimensional vector with zero in w.
  * @relates vec4
  */
vec4 avec4 (float x, float y, float z)
{
    return vec4.create (x, y, z);
}

/** Create a four-dimensional vector.
  * @relates vec4
  */
vec4 avec4 (float x, float y, float z, float w)
{
    return vec4.create (x, y, z, w);
}

/** Create a four-dimensional vector by zero-extending a three-dimensional vector.
  * @relates vec4
  */
vec4 avec4 (vec3 base)
{
    return vec4.create (base.x, base.y, base.z, 0);
}

/** A 2x2 matrix. */
struct mat2
{
    vec2 cols [2]; /**< The matrix data, in OpenGL's silly order (transposed from sensible). */

    /** Create a matrix using normal order. */
    static mat2 create (vec2 a, vec2 b)
    {
        mat2 r;

        r.row (0, a);
        r.row (1, b);

        return r;
    }

    /** Create a matrix using normal order. */
    static mat2 create (float aa, float ab,
                        float ba, float bb)
    {
        mat2 r;

        r.row (0, aa, ab);
        r.row (1, ba, bb);

        return r;
    }

    /** Set this matrix. */
    void set (vec2 a, vec2 b)
    {
        row (0, a);
        row (1, b);
    }

    /** Set this matrix. */
    void set (float aa, float ab,
              float ba, float bb)
    {
        row (0, aa, ab);
        row (1, ba, bb);
    }

    /** Get the identity matrix. */
    static mat2 identity ()
    {
        return create (1, 0, 0, 1);
    }

    /** Get a scaling matrix. */
    static mat2 scaling (float x)
    {
        return create (x, 0,
                       0, 1);
    }

    /** Get a scaling matrix. */
    static mat2 scaling (vec2 amount)
    {
        return create (amount.x, 0,
                       0, amount.y);
    }

    /** Get a translation matrix. */
    static mat2 translation (float amount)
    {
        return create (1, amount, 0, 1);
    }

    /** Multiply by a scaling matrix. */
    mat2 scale (float x)
    {
        return *this * scaling (x);
    }

    /** Multiply by a scaling matrix. */
    mat2 scale (vec2 amount)
    {
        return *this * scaling (amount);
    }

    /** Multiply by a translation matrix. */
    mat2 translate (float x)
    {
        return *this * translation (x);
    }

    /** Return whether this is an identity matrix. */
    bit isIdentity ()
    {
        return cols [0] == vec2.create (1, 0)
            && cols [1] == vec2.create (0, 1);
    }

    /** Set a cell on the matrix. */
    void set (int col, int row, float value)
    {
        cols [col].index (row, value);
    }

    /** Get a cell on the matrix. */
    float get (int col, int row)
    {
        return cols [col].index (row);
    }

    /** Set a row on the matrix. */
    void row (int row, float a, float b)
    {
        set (0, row, a);
        set (1, row, b);
    }

    /** Set a row on the matrix. */
    void row (int row, vec2 vec)
    {
        set (0, row, vec.x);
        set (1, row, vec.y);
    }

    /** Get a column on the matrix. */
    vec2 col (int col)
    {
        return vec2.create (cols [0].index (col), cols [1].index (col));
    }

    /** Get a column on the matrix as a scalar. */
    float col1 (int col)
    {
        return cols [0].index (col);
    }

    /** Set a column on the matrix. */
    void col (int col, float a, float b)
    {
        set (col, 0, a);
        set (col, 1, b);
    }

    /** Set a column on the matrix. */
    void col (int col, vec2 vec)
    {
        set (col, 0, vec.x);
        set (col, 1, vec.y);
    }

    /** Get the diagonal (which encodes the simple scale). */
    vec2 diag ()
    {
        return vec2.create (get (0, 0), get (1, 1));
    }

    /** Get the full 4-cell array. */
    float [] array () { return (&cols [0].x) [0 .. 4]; }

    /** Swizzle */
    mat2 swizzle (int a, int b) { return create (cols [a], cols [b]); }

    /** Rotate the scalar, ignoring the y component. */
    float rotate (float v)
    {
        return cols [0].x * v;
    }

    /** Get the transpose matrix. */
    mat2 transpose ()
    {
        mat2 r;
    
        r.row (0, cols [0].x, cols [0].y);
        r.row (1, cols [1].x, cols [1].y);
        return r;
    }

    /** Multiply matrices. */
    mat2 opMul (mat2 mb)
    {
        mat2 mo;
        float [] a = array ();
        float [] b = mb.array ();
        float [] o = mo.array ();

        for (int i; i < 2; i ++)
        {
            o [i + 0] = a [i] * b [0] + a [i + 2] * b [1];
            o [i + 1] = a [i] * b [2] + a [i + 2] * b [3];
        }

        return mo;
    }

    /** Multiply matrix by one-extended scalar. */
    float opMul (float v)
    {
        return cols [0].x * v + cols [1].x;
    }

    /** Multiply matrix by vector. */
    vec2 opMul (vec2 v)
    {
        return v.create (cols [0].x * v.x + cols [1].x * v.y,
                         cols [0].y * v.x + cols [1].y * v.y);
    }

    /** Test for equality. */
     bit opEqual (mat2 b)
    {
        return cols [0] == b.cols [0] && cols [1] == b.cols [1];
    }

    /** Print the matrix. */
    void print ()
    {
        printf ("%.*s\n", toString ());
    }

    /** Convert to a four-line std.string. */
    char [] toString ()
    {
        char [4096] buffer;
        int length;
        int a, b;

        a = imax (std.c.stdio.sprintf (buffer, "%g", get (0, 0)), std.c.stdio.sprintf (buffer, "%g", get (0, 1)));
        b = imax (std.c.stdio.sprintf (buffer, "%g", get (1, 0)), std.c.stdio.sprintf (buffer, "%g", get (1, 1)));

        length = 
        std.c.stdio.sprintf (buffer, 
            "[ %*g %*g ]\n"
            "[ %*g %*g ]\n",
            a, get (0, 0), b, get (1, 0),
            a, get (0, 1), b, get (1, 1),
            a, get (0, 2), b, get (1, 2));

        return buffer [0 .. length].dup;
    }
}

/** A 3x3 matrix. */
struct mat3
{
/+
#ifdef DoxygenShouldSkipThis
+/

    vec3 cols [3] = [{1, 0, 0}, {0, 1, 0}, {0, 0, 1}]; /**< The matrix data, in OpenGL's silly order (transposed from sensible). */
    
/+
#endif
+/

    /** Create a rotation matrix that rotates a vector from one point to another point (stolen from OPCODE). */
    static mat3 fromTo (vec3 from, vec3 to)
    {
        from = from.normalize ();
        to = to.normalize ();
        vec3 v = from.cross (to);
        float e = from.dot (to);
        mat3 result;

        /* "from" almost equal to "to". */
        if (e > 1 - 0.0001)
            return mat3.identity ();
        /* "from" almost equal to negated "to". */
        else if (e <  -1 + 0.0001)
        {
            vec3 left = avec3 (0, from.z, -from.y);

            if (left.dot (left) < 0.0001)
                left = avec3 (-from.z, 0, from.x);
            left = left.normalize ();

            vec3 up = left.cross (from);

	    /* now we have a coordinate system, i.e., a basis;    */
	    /* M=(from, up, left), and we want to rotate to:      */
	    /* N=(-from, up, -left). This is done with the matrix:*/
	    /* N*M^T where M^T is the transpose of M              */
	    float fxx=-from.x*from.x; float fyy=-from.y*from.y; float fzz=-from.z*from.z;
	    float fxy=-from.x*from.y; float fxz=-from.x*from.z; float fyz=-from.y*from.z;

	    float uxx=up.x*up.x; float uyy=up.y*up.y; float uzz=up.z*up.z;
	    float uxy=up.x*up.y; float uxz=up.x*up.z; float uyz=up.y*up.z;

	    float lxx=-left.x*left.x; float lyy=-left.y*left.y; float lzz=-left.z*left.z;
	    float lxy=-left.x*left.y; float lxz=-left.x*left.z; float lyz=-left.y*left.z;
	    // symmetric matrix
	    result.set (0, 0, fxx+uxx+lxx);	result.set (1, 0, fxy+uxy+lxy);	result.set (2, 0, fxz+uxz+lxz);
	    result.set (0, 1, result.get (1, 0));		result.set (1, 1, fyy+uyy+lyy);	result.set (2, 1, fyz+uyz+lyz);
	    result.set (0, 2, result.get (2, 0));		result.set (1, 2, result.get (2, 1));		result.set (2, 2, fzz+uzz+lzz);
        }
        else
        {
	    float h= (1.0f - e) / v.dot (v);
	    float hvx  = h*v.x;
	    float hvz  = h*v.z;
	    float hvxy = hvx*v.y;
	    float hvxz = hvx*v.z;
	    float hvyz = hvz*v.y;
	    result.set (0, 0, e+hvx*v.x);	result.set (1, 0, hvxy-v.z);		result.set (2, 0, hvxz+v.y);
	    result.set (0, 1, hvxy+v.z);	result.set (1, 1, e+h*v.y*v.y);	result.set (2, 1, hvyz-v.x);
	    result.set (0, 2, hvxz-v.y);	result.set (1, 2, hvyz+v.x);		result.set (2, 2, e+hvz*v.z);
        }

        return result.transpose ();
    }

    /** Create a rotation matrix from a normal.  The normal need not be normalised. */
    static mat3 fromNormal (vec3 normal)
    {
        if (normal.y >= 0.9999)
            return mat3.rotationDeg (90, 1, 0, 0);
        if (normal.y <= -0.9999)
            return mat3.rotationDeg (-90, 1, 0, 0);
        return fromTo (avec3 (1, 0, 0), avec3 (normal.z, -normal.y, -normal.x));
    }

    /** Create a matrix using normal order. */
    static mat3 create (vec3 a, vec3 b, vec3 c)
    {
        mat3 r;

        r.row (0, a);
        r.row (1, b);
        r.row (2, c);

        return r;
    }

    /** Create a matrix using normal order. */
    static mat3 create (float aa, float ab, float ac,
                        float ba, float bb, float bc,
                        float ca, float cb, float cc)
    {
        mat3 r;

        r.row (0, aa, ab, ac);
        r.row (1, ba, bb, bc);
        r.row (2, ca, cb, cc);

        return r;
    }

    /** Set this matrix. */
    void set (vec3 a, vec3 b, vec3 c)
    {
        row (0, a);
        row (1, b);
        row (2, c);
    }

    /** Set this matrix. */
    void set (float aa, float ab, float ac,
              float ba, float bb, float bc,
              float ca, float cb, float cc)
    {
        row (0, aa, ab, ac);
        row (1, ba, bb, bc);
        row (2, ca, cb, cc);
    }

    /** Get the identity matrix. */
    static mat3 identity ()
    {
        return create (1, 0, 0,
                       0, 1, 0,
                       0, 0, 1);
    }

    /** Get a rotation matrix.  angle is in degrees. */
    static mat3 rotationDeg (float angle, float x, float y, float z)
    {
        return rotationDeg (angle, vec3.create (x, y, z));
    }

    /** Get a rotation matrix.  angle is in degrees. */
    static mat3 rotationDeg (float angle, vec3 v)
    {
        v = v.normalize ();
        float c = std.math.cos (angle * std.math.PI / 180);
        float s = std.math.sin (angle * std.math.PI / 180);
        float x = v.x, y = v.y, z = v.z;

        mat3 mat;

        mat.row (0, x * x * (1 - c) + c, x * y * (1 - c) - z * s, x * z * (1 - c) + y * s);
        mat.row (1, y * x * (1 - c) + z * s, y * y * (1 - c) + c, y * z * (1 - c) - x * s);
        mat.row (2, x * z * (1 - c) - y * s, y * z * (1 - c) + x * s, z * z * (1 - c) + c);

        return mat;
    }

    /** Get a rotation matrix.  angle is in radians. */
    static mat3 rotationRad (float angle, float x, float y, float z)
    {
        return rotationDeg (angle * 180 / std.math.PI, vec3.create (x, y, z));
    }

    /** Get a rotation matrix.  angle is in radians. */
    static mat3 rotationRad (float angle, vec3 v)
    {
        return rotationDeg (angle * 180 / std.math.PI, v);
    }

    /** Get a scaling matrix. */
    static mat3 scaling (float x, float y)
    {
        return create (x, 0, 0,
                       0, y, 0,
                       0, 0, 1);
    }

    /** Get a scaling matrix. */
    static mat3 scaling (vec2 amount)
    {
        return create (amount.x, 0, 0,
                       0, amount.y, 0,
                       0, 0, 1);
    }

    /** Get a scaling matrix. */
    static mat3 scaling (vec3 amount)
    {
        return create (amount.x, 0, 0,
                       0, amount.y, 0,
                       0, 0, amount.z);
    }

    /** Get a translation matrix. */
    static mat3 translation (float x, float y)
    {
        return create (1, 0, x,
                       0, 1, y,
                       0, 0, 1);
    }

    /** Get a translation matrix. */
    static mat3 translation (vec2 v)
    {
        return create (1, 0, v.x,
                       0, 1, v.y,
                       0, 0, 1);
    }

    /** Multiply by a scaling matrix. */
    mat3 scale (float x, float y)
    {
        return *this * scaling (x, y);
    }

    /** Multiply by a scaling matrix. */
    mat3 scale (vec2 amount)
    {
        return *this * scaling (amount);
    }

    /** Multiply by a scaling matrix. */
    mat3 scale (vec3 amount)
    {
        return *this * scaling (amount);
    }

    /** Multiply by a translation matrix. */
    mat3 translate (float x, float y)
    {
        return *this * translation (x, y);
    }

    /** Multiply by a translation matrix. */
    mat3 translate (vec2 amount)
    {
        return *this * translation (amount);
    }

    /** Return whether this is an identity matrix. */
    bit isIdentity ()
    {
        return cols [0] == vec3.create (1, 0, 0)
            && cols [1] == vec3.create (0, 1, 0)
            && cols [2] == vec3.create (0, 0, 1);
    }

    /** Set a cell on the matrix. */
    void set (int col, int row, float value)
    {
        cols [col].index (row, value);
    }

    /** Get a cell on the matrix. */
    float get (int col, int row)
    {
        return cols [col].index (row);
    }

    /** Get the diagonal (which encodes the simple scale). */
    vec3 diag ()
    {
        return vec3.create (get (0, 0), get (1, 1), get (2, 2));
    }

    /** Set a row on the matrix. */
    void row (int row, float a, float b, float c)
    {
        set (0, row, a);
        set (1, row, b);
        set (2, row, c);
    }

    /** Set a row on the matrix. */
    void row (int row, vec3 vec)
    {
        set (0, row, vec.x);
        set (1, row, vec.y);
        set (2, row, vec.z);
    }

    /** Return a row on the matrix. */
    vec3 row (int row)
    {
        return avec3 (get (0, row), get (1, row), get (2, row));
    }

    /** Get a column on the matrix. */
    vec3 col (int col)
    {
        return vec3.create (get (col, 0), get (col, 1), get (col, 2));
    }

    /** Get a column on the matrix as a vec2. */
    vec2 col2 (int col)
    {
        return vec2.create (get (col, 0), get (col, 1));
    }

    /** Set a column on the matrix. */
    void col (int col, float a, float b, float c)
    {
        set (col, 0, a);
        set (col, 1, b);
        set (col, 2, c);
    }

    /** Set a column on the matrix. */
    void col (int col, vec3 vec)
    {
        set (col, 0, vec.x);
        set (col, 1, vec.y);
        set (col, 2, vec.z);
    }

    /** Get the full 9-cell array. */
    float [] array () { return (&cols [0].x) [0 .. 9]; }

    /** Swizzle */
    mat3 swizzle (int a, int b, int c) { return create (cols [a], cols [b], cols [c]); }

    /** Rotate the vector, ignoring the z component. */
    vec2 rotate (vec2 v)
    {
        return v.create (cols [0].x * v.x + cols [1].x * v.y,
                         cols [0].y * v.x + cols [1].y * v.y);
    }

    /** Get the transpose matrix. */
    mat3 transpose ()
    {
        mat3 r;
    
        r.row (0, cols [0].x, cols [0].y, cols [0].z);
        r.row (1, cols [1].x, cols [1].y, cols [1].z);
        r.row (2, cols [2].x, cols [2].y, cols [2].z);
        return r;
    }

    /** Multiply matrices. */
    mat3 opMul (mat3 mb)
    {
        mat3 mo;
        float [] a = array ();
        float [] b = mb.array ();
        float [] o = mo.array ();

        for (int i; i < 3; i ++)
        {
            o [i + 0] = a [i] * b [0] + a [i + 3] * b [1] + a [i + 6] * b [2];
            o [i + 3] = a [i] * b [3] + a [i + 3] * b [4] + a [i + 6] * b [5];
            o [i + 6] = a [i] * b [6] + a [i + 3] * b [7] + a [i + 6] * b [8];
        }

        return mo;
    }

    /** Multiply matrix by one-extended vector. */
    vec2 opMul (vec2 v)
    {
        return v.create (cols [0].x * v.x + cols [1].x * v.y + cols [2].x,
                         cols [0].y * v.x + cols [1].y * v.y + cols [2].y);
    }

    /** Multiply matrix by vector. */
    vec3 opMul (vec3 v)
    {
        return avec3 (cols [0].x * v.x + cols [1].x * v.y + cols [2].x * v.z,
                         cols [0].y * v.x + cols [1].y * v.y + cols [2].y * v.z,
                         cols [0].z * v.x + cols [1].z * v.y + cols [2].z * v.z);
    }

    /** Multiply and assign matrix. */
    mat3 opMulAssign (mat3 mb)
    {
        return *this = *this * mb;
    }

    /** Test for equality. */
     bit opEqual (mat3 b)
    {
        return cols [0] == b.cols [0] && cols [1] == b.cols [1] && cols [2] == b.cols [2];
    }

    /** Print the matrix. */
    void print ()
    {
        printf ("%.*s\n", toString ());
    }

    private float mget (int col, int row)
    {
        float value = get (col, row);

        if (std.math.fabs (value) < 0.001)
            return 0;
        return value;
    }

    /** Convert to a four-line std.string. */
    char [] toString ()
    {
        char [4096] buffer;
        int a, b, c;
        int length;

        a = imax (std.c.stdio.sprintf (buffer, "%g", mget (0, 0)), std.c.stdio.sprintf (buffer, "%g", mget (0, 1)), std.c.stdio.sprintf (buffer, "%g", mget (0, 2)));
        b = imax (std.c.stdio.sprintf (buffer, "%g", mget (1, 0)), std.c.stdio.sprintf (buffer, "%g", mget (1, 1)), std.c.stdio.sprintf (buffer, "%g", mget (1, 2)));
        c = imax (std.c.stdio.sprintf (buffer, "%g", mget (2, 0)), std.c.stdio.sprintf (buffer, "%g", mget (2, 1)), std.c.stdio.sprintf (buffer, "%g", mget (2, 2)));

        length = 
        std.c.stdio.sprintf (buffer, 
            "[ %*g %*g %*g ]\n"
            "[ %*g %*g %*g ]\n"
            "[ %*g %*g %*g ]\n",
            a, mget (0, 0), b, mget (1, 0), c, mget (2, 0),
            a, mget (0, 1), b, mget (1, 1), c, mget (2, 1),
            a, mget (0, 2), b, mget (1, 2), c, mget (2, 2));

        return buffer [0 .. length].dup;
    }
}

/** A 4x4 matrix. */
struct mat4
{
/+
#ifdef DOXYGEN_WONT_LIKE_THIS
+/
    float m [4] [4] = [ [ 1, 0, 0, 0 ], [ 0, 1, 0, 0 ], [ 0, 0, 1, 0 ], [ 0, 0, 0, 1 ] ]; /**< The matrix data, in OpenGL's silly order (transposed from sensible). */
/+
#endif
+/

    /** Create a matrix using normal order. */
    static mat4 create (vec4 a, vec4 b, vec4 c, vec4 d)
    {
        mat4 r;

        r.row (0, a);
        r.row (1, b);
        r.row (2, c);
        r.row (3, d);

        return r;
    }

    /** Create a matrix. */
    static mat4 create (float aa, float ab, float ac, float ad, 
                        float ba, float bb, float bc, float bd, 
                        float ca, float cb, float cc, float cd, 
                        float da, float db, float dc, float dd) 
    {
        return create (vec4.create (aa, ab, ac, ad), 
                       vec4.create (ba, bb, bc, bd),
                       vec4.create (ca, cb, cc, cd),
                       vec4.create (da, db, dc, dd));
    }

    /** Create a matrix by extending a 3x3 matrix with identity. */
    static mat4 create (mat3 base)
    {
        return create (avec4 (base.cols [0]), avec4 (base.cols [1]), avec4 (base.cols [2]), avec4 (0, 0, 0, 1));

    }

    /** Set this matrix. */
    void set (vec4 a, vec4 b, vec4 c, vec4 d)
    {
        row (0, a);
        row (1, b);
        row (2, c);
        row (3, d);
    }

    /** Set this matrix. */
    void set (float aa, float ab, float ac, float ad, 
              float ba, float bb, float bc, float bd, 
              float ca, float cb, float cc, float cd, 
              float da, float db, float dc, float dd) 
    {
        row (0, aa, ab, ac, ad);
        row (1, ba, bb, bc, bd);
        row (2, ca, cb, cc, cd);
        row (3, da, db, dc, dd);
    }

    /** Get the identity matrix. */
    static mat4 identity ()
    {
        return create (1, 0, 0, 0,
                       0, 1, 0, 0,
                       0, 0, 1, 0,
                       0, 0, 0, 1);
    }

    /** Get a scaling matrix. */
    static mat4 scaling (float x, float y, float z)
    {
        return create (x, 0, 0, 0,
                       0, y, 0, 0,
                       0, 0, z, 0,
                       0, 0, 0, 1);
    }

    /** Get a scaling matrix. */
    static mat4 scaling (vec3 amount)
    {
        return create (amount.x, 0, 0, 0,
                       0, amount.y, 0, 0,
                       0, 0, amount.z, 0,
                       0, 0, 0, 1);
    }

    /** Get a scaling matrix. */
    static mat4 scaling (vec4 amount)
    {
        return create (amount.x, 0, 0, 0,
                       0, amount.y, 0, 0,
                       0, 0, amount.z, 0,
                       0, 0, 0, amount.w);
    }

    /** Get a translation matrix. */
    static mat4 translation (float x, float y, float z)
    {
        return create (1, 0, 0, x,
                       0, 1, 0, y,
                       0, 0, 1, z,
                       0, 0, 0, 1);
    }

    /** Get a translation matrix. */
    static mat4 translation (vec3 v)
    {
        return create (1, 0, 0, v.x,
                       0, 1, 0, v.y,
                       0, 0, 1, v.z,
                       0, 0, 0, 1);
    }

    /** Get a rotation matrix.  angle is in degrees. */
    static mat4 rotationDeg (float angle, float x, float y, float z)
    {
        return rotationDeg (angle, vec3.create (x, y, z));
    }

    /** Get a rotation matrix.  angle is in degrees. */
    static mat4 rotationDeg (float angle, vec3 v)
    {
        v = v.normalize ();
        float c = std.math.cos (angle * std.math.PI / 180);
        float s = std.math.sin (angle * std.math.PI / 180);
        float x = v.x, y = v.y, z = v.z;

        mat4 mat;

        mat.row (0, x * x * (1 - c) + c, x * y * (1 - c) - z * s, x * z * (1 - c) + y * s, 0);
        mat.row (1, y * x * (1 - c) + z * s, y * y * (1 - c) + c, y * z * (1 - c) - x * s, 0);
        mat.row (2, x * z * (1 - c) - y * s, y * z * (1 - c) + x * s, z * z * (1 - c) + c, 0);
        mat.row (3, 0, 0, 0, 1);

        return mat;
    }

    /** Multiply by a scaling matrix. */
    mat4 scale (float x, float y, float z)
    {
        return *this * scaling (x, y, z);
    }

    /** Multiply by a scaling matrix. */
    mat4 scale (vec3 amount)
    {
        return *this * scaling (amount);
    }

    /** Multiply by a scaling matrix. */
    mat4 scale (vec4 amount)
    {
        return *this * scaling (amount);
    }

    /** Multiply by a translation matrix. */
    mat4 translate (float x, float y, float z)
    {
        return *this * translation (x, y, z);
    }

    /** Multiply by a translation matrix. */
    mat4 translate (vec3 amount)
    {
        return *this * translation (amount);
    }

    /** Multiply by a rotation matrix.  angle is in degrees. */
    mat4 rotateDeg (float angle, float x, float y, float z)
    {
        return *this * rotationDeg (angle, x, y, z);
    }

    /** Multiply by a rotation matrix.  angle is in degrees. */
    mat4 rotateDeg (float angle, vec3 v)
    {
        return *this * rotationDeg (angle, v);
    }

    /** Return whether this is an identity matrix. */
    bit isIdentity ()
    {
        return row (0) == vec4.create (1, 0, 0, 0)
            && row (1) == vec4.create (0, 1, 0, 0)
            && row (2) == vec4.create (0, 0, 1, 0)
            && row (3) == vec4.create (0, 0, 0, 1);
    }

    /** Set a cell on the matrix. */
    void set (int row, int col, float value)
    {
        m [col] [row] = value;
        //cols [col].index (row, value);
    }

    /** Get a cell on the matrix. */
    float get (int row, int col)
    {
        return m [col] [row];
        //return cols [col].index (row);
    }

    /** Set a row on the matrix. */
    void row (int row, float a, float b, float c, float d)
    {
        set (row, 0, a);
        set (row, 1, b);
        set (row, 2, c);
        set (row, 3, d);
    }

    /** Set a row on the matrix. */
    void row (int row, vec4 vec)
    {
        set (row, 0, vec.x);
        set (row, 1, vec.y);
        set (row, 2, vec.z);
        set (row, 3, vec.w);
    }

    /** Get a row on the matrix. */
    vec4 row (int row)
    {
        return vec4.create (get (row, 0), get (row, 1), get (row, 2), get (row, 3));
    }

    /** Get a column on the matrix. */
    vec4 col (int col)
    {
        return vec4.create (get (0, col), get (1, col), get (2, col), get (3, col));
    }

    /** Get a column on the matrix as a vec3. */
    vec3 col3 (int col)
    {
        return vec3.create (get (0, col), get (1, col), get (2, col));
    }

    /** Set a column on the matrix. */
    void col (int col, float a, float b, float c, float d)
    {
        set (0, col, a);
        set (1, col, b);
        set (2, col, c);
        set (3, col, d);
    }

    /** Set a column on the matrix. */
    void col (int col, vec4 vec)
    {
        set (0, col, vec.x);
        set (1, col, vec.y);
        set (2, col, vec.z);
        set (3, col, vec.w);
    }

    /** Get the diagonal (which encodes the simple scale). */
    vec4 diag ()
    {
        return vec4.create (get (0, 0), get (1, 1), get (2, 2), get (3, 3));
    }

    /** Get the full 16-cell array. */
    float [] array () { return ((float *) m) [0 .. 16]; }

    /** Swizzle */
    mat4 swizzle (int a, int b, int c, int d) { return create (col (a), col (b), col (c), col (d)); }

    /** Rotate the vector, ignoring the w component. */
    vec3 rotate (vec3 v)
    {
        return avec3 (m [0] [0] * v.x + m [0] [1] * v.y + m [0] [2] * v.z,
                         m [1] [0] * v.x + m [1] [1] * v.y + m [1] [2] * v.z,
                         m [2] [0] * v.x + m [2] [1] * v.y + m [2] [2] * v.z);
    }

    /** Rotate the vector, ignoring the w component, and renormalize the vector if necessary. */
    vec3 rotateNormal (vec3 v)
    {
        return rotate (v).normalize ();
    }

    /** Get the transpose matrix. */
    mat4 transpose ()
    {
        mat4 r;
    
        r.row (0, col (0));
        r.row (1, col (1));
        r.row (2, col (2));
        r.row (3, col (3));
        return r;
    }

    /** Get whether the parity is positive. */
    bit parity ()
    {
        vec3 col0 = col3 (0);
        vec3 col1 = col3 (1);
        vec3 col2 = col3 (2);

        return col0.cross (col1).dot (col2) >= 0;
    }

    /** Multiply by scalar. */
    mat4 opMul (float v)
    {
        return mat4.create (row (0) * v, row (1) * v, row (2) * v, row (3) * v);
    }

    /** Multiply matrices. */
    mat4 opMul (mat4 mb)
    {
        mat4 mo;
        float [] a = array ();
        float [] b = mb.array ();
        float [] o = mo.array ();

        for (int i; i < 4; i ++)
        {
            o [i] = a [i] * b [0] + a [i + 4] * b [1] + a [i + 8] * b [2] + a [i + 12] * b [3];
            o [i + 4] = a [i] * b [4] + a [i + 4] * b [5] + a [i + 8] * b [6] + a [i + 12] * b [7];
            o [i + 8] = a [i] * b [8] + a [i + 4] * b [9] + a [i + 8] * b [10] + a [i + 12] * b [11];
            o [i + 12] = a [i] * b [12] + a [i + 4] * b [13] + a [i + 8] * b [14] + a [i + 12] * b [15];
        }

        return mo;
    }

    /** Multiply matrix by zero and one-extended vector. */
    vec2 opMul (vec2 v)
    {
        return v.create (m [0] [0] * v.x + m [1] [0] * v.y + m [3] [0],
                         m [0] [1] * v.x + m [1] [1] * v.y + m [3] [1]);
    }

    /** Multiply matrix by one-extended vector. */
    vec3 opMul (vec3 v)
    {
        return avec3 (m [0] [0] * v.x + m [1] [0] * v.y + m [2] [0] * v.z + m [3] [0],
                         m [0] [1] * v.x + m [1] [1] * v.y + m [2] [1] * v.z + m [3] [1],
                         m [0] [2] * v.x + m [1] [2] * v.y + m [2] [2] * v.z + m [3] [2]);
    }

    /** Multiply matrix by vector. */
    vec4 opMul (vec4 v)
    {
        return avec4 (m [0] [0] * v.x + m [1] [0] * v.y + m [2] [0] * v.z + m [3] [0] * v.w,
                         m [0] [1] * v.x + m [1] [1] * v.y + m [2] [1] * v.z + m [3] [1] * v.w,
                         m [0] [2] * v.x + m [1] [2] * v.y + m [2] [2] * v.z + m [3] [2] * v.w,
                         m [0] [3] * v.x + m [1] [3] * v.y + m [2] [3] * v.z + m [3] [3] * v.w);
    }

    /** Compute the cofactor of a cell, stolen from Pierre Terdiman. */
    float cofactor (int row, int col)
    {
        int r1 = (row + 1) & 3;
        int r2 = (row + 2) & 3;
        int r3 = (row + 3) & 3;
        int c1 = (col + 1) & 3;
        int c2 = (col + 2) & 3;
        int c3 = (col + 3) & 3;
        float r;

        r = ((m [r1] [c1] * m [r2] [c2] * m [r3] [c3]
            + m [r1] [c2] * m [r2] [c3] * m [r3] [c1]
            + m [r1] [c3] * m [r2] [c1] * m [r3] [c2])
           - (m [r3] [c1] * m [r2] [c2] * m [r1] [c3]
            + m [r3] [c2] * m [r2] [c3] * m [r1] [c1]
            + m [r3] [c3] * m [r2] [c1] * m [r1] [c2]));
        
        if ((row + col) & 1)
            return -r;
        return r;
    }

    /** Compute a matrix filled with cofactors of this matrix. */
    mat4 cofactorMatrix ()
    {
        mat4 o;

        for (int c; c < 4; c ++)
        for (int r; r < 4; r ++)
            o.set (r, c, cofactor (r, c));

        return o;
    }

    /** Compute the determinant of this matrix. */
    float determinant ()
    {
        return m [0] [0] * cofactor (0, 0)
             + m [0] [1] * cofactor (0, 1)
             + m [0] [2] * cofactor (0, 2)
             + m [0] [3] * cofactor (0, 3);
    }

    /** Invert matrix, throws if std.math.singular. */
    mat4 invert ()
    {
        float det = determinant ();
        mat4 o;

        if (digCommonFloatAbs (det) < 1.0e-7)
            throw new Error ("Matrix is std.math.singular and cannot be inverted.");

        det = 1.0 / det;

        o.m [0] [0] = cofactor (0, 0) * det;
        o.m [1] [0] = cofactor (0, 1) * det;
        o.m [2] [0] = cofactor (0, 2) * det;
        o.m [3] [0] = cofactor (0, 3) * det;

        o.m [0] [1] = cofactor (1, 0) * det;
        o.m [1] [1] = cofactor (1, 1) * det;
        o.m [2] [1] = cofactor (1, 2) * det;
        o.m [3] [1] = cofactor (1, 3) * det;

        o.m [0] [2] = cofactor (2, 0) * det;
        o.m [1] [2] = cofactor (2, 1) * det;
        o.m [2] [2] = cofactor (2, 2) * det;
        o.m [3] [2] = cofactor (2, 3) * det;

        o.m [0] [3] = cofactor (3, 0) * det;
        o.m [1] [3] = cofactor (3, 1) * det;
        o.m [2] [3] = cofactor (3, 2) * det;
        o.m [3] [3] = cofactor (3, 3) * det;

        return o;
    }

    /** Return the rotation and scaling factors of the matrix. */
    mat4 rotationOnly ()
    {
        mat4 result = *this;

        result.col (3, 0, 0, 0, 1);
        result.row (3, 0, 0, 0, 1);
        return result;
    }

    /** Test for equality. */
     bit opEqual (mat4 b)
    {
        for (int c; c < 4; c ++)
            for (int r; r < 4; r ++)
                if (m [r] [c] != b.m [r] [c])
                    return false;

        return true;
    }

    /** Print the matrix. */
    void print ()
    {
        printf ("%.*s\n", toString ());
    }

    /** Convert to a four-line std.string. */
    char [] toString ()
    {
        char [4096] buffer;
        int a, b, c, d;
        int length;

        a = imax (std.c.stdio.sprintf (buffer, "%g", get (0, 0)), std.c.stdio.sprintf (buffer, "%g", get (1, 0)), std.c.stdio.sprintf (buffer, "%g", get (2, 0)), std.c.stdio.sprintf (buffer, "%g", get (3, 0)));
        b = imax (std.c.stdio.sprintf (buffer, "%g", get (0, 1)), std.c.stdio.sprintf (buffer, "%g", get (1, 1)), std.c.stdio.sprintf (buffer, "%g", get (2, 1)), std.c.stdio.sprintf (buffer, "%g", get (3, 1)));
        c = imax (std.c.stdio.sprintf (buffer, "%g", get (0, 2)), std.c.stdio.sprintf (buffer, "%g", get (1, 2)), std.c.stdio.sprintf (buffer, "%g", get (2, 2)), std.c.stdio.sprintf (buffer, "%g", get (3, 2)));
        d = imax (std.c.stdio.sprintf (buffer, "%g", get (0, 3)), std.c.stdio.sprintf (buffer, "%g", get (1, 3)), std.c.stdio.sprintf (buffer, "%g", get (2, 3)), std.c.stdio.sprintf (buffer, "%g", get (3, 3)));

        length = 
        std.c.stdio.sprintf (buffer, 
            "[ %*g %*g %*g %*g ]\n"
            "[ %*g %*g %*g %*g ]\n"
            "[ %*g %*g %*g %*g ]\n"
            "[ %*g %*g %*g %*g ]\n", 
            a, get (0, 0), b, get (0, 1), c, get (0, 2), d, get (0, 3),
            a, get (1, 0), b, get (1, 1), c, get (1, 2), d, get (1, 3),
            a, get (2, 0), b, get (2, 1), c, get (2, 2), d, get (2, 3),
            a, get (3, 0), b, get (3, 1), c, get (3, 2), d, get (3, 3));

        return buffer [0 .. length].dup;
    }

    /** Return the quaternion component from the matrix. */
    quat3 toquat3 ()
    {
        quat3 quat;
        const int [3] next = [ 1, 2, 0 ];
        float diag, s;
        float [4] o;
        int i, j, k;
        
        diag = get (0, 0) + get (1, 1) + get (2, 2);
        
        if (diag > 0)
        {
            s = std.math.sqrt (diag + 1);
            quat.w = s / 2;
            s = 0.5 / s;
            quat.x = (get (1, 2) - get (2, 1)) * s;
            quat.y = (get (2, 0) - get (0, 2)) * s;
            quat.z = (get (0, 1) - get (1, 0)) * s;
        }
        else
        {
            i = 0;
            if (get (1, 1) > get (0, 0))
                i = 1;
            if (get (2, 2) > get (i, i))
                i = 2;
            j = next [i];
            k = next [j];
            s = get (i, i) - (get (j, j) + get (k, k));
            s = std.math.sqrt (s) + 1;
            
            o [i] = s / 2;
            s = 0.5 / s;
            o [j] = (get (i, j) + get (j, i)) * s;
            o [k] = (get (i, k) + get (k, i)) * s;
            o [3] = (get (j, k) - get (k, j)) * s;
            
            quat.x = o [0];
            quat.y = o [1];
            quat.z = o [2];
            quat.w = o [3];
        }

        return quat;
    }
}

unittest
{
    mat4 identity = mat4.identity ();
    mat4 everywhere = mat4.create (6, 32, 543, 21, 
                                   654, 213, 83, 12,
                                   9, -432, -3, 49,
                                   34, 96.4, 48, -2);

    assert (identity.isIdentity ());
    assert ((identity * identity).isIdentity ());

    assert (mat4.scaling (1, 2, 3) == mat4.create (1, 0, 0, 0,
                                                   0, 2, 0, 0,
                                                   0, 0, 3, 0,
                                                   0, 0, 0, 1));

    mat4 m = mat4.scaling (1, 2, 3).translate (1, 2, 3);
    vec3 v = vec3.create (1, 1, 1), w;

    w = m * v;
    
    assert (m
         == mat4.create (1, 0, 0, 1,
                         0, 2, 0, 4,
                         0, 0, 3, 9,
                         0, 0, 0, 1));
}

/** A two-dimensional bounding box. */
struct box2
{
    vec2 min; /**< Starting bounds. */
    vec2 max; /**< Ending bounds. */
    
    /** Create the box with a specific value for the starting and ending bounds. */
    static box2 create (vec2 vector)
    {
        return create (vector, vector);
    }
    
    /** Create a box with specific values. */
    static box2 create (vec2 min, vec2 max)
    {
        box2 result;
        
        result.set (min, max);
        return result;
    }
    
    /** Create a box by specifying the center and the length of each side. */
    static box2 fromCenter (vec2 center, vec2 length)
    {
        length = length.abs ();
        length = length / 2;
        return create (center - length, center + length);
    }
    
    /** Assign specific values. */
    void set (vec2 min, vec2 max)
    {
        this.min = min;
        this.max = max;
    }
    
    /** Return the center of the box. */
    vec2 center ()
    {
        return (min + max) / 2;
    }
    
    /** Change the center of the box. */
    void center (vec2 value)
    {
        *this += value - center ();
    }
    
    /** Return whether this point is conservatively within the box. */
    bit concontains (vec2 point)
    {
        if (isnan ())
            return false;
        return point.cmpg (min) && point.cmpl (max);
    }

    /** Return whether this box conservatively envelopes b (greater than rather than greater than or equal to) .*/
    bit concontains (box2 box)
    {
        if (isnan () || box.isnan ())
            return false;
        return box.min.cmpg (min) && box.max.cmpl (max);
    }

    /** Return whether the box conservatively overlaps the other box. */
    bit conoverlaps (box2 box)
    {
        if (isnan () || box.isnan ())
            return false;
        return box.max.cmpg (min) && box.min.cmpl (max);
    }

    /** Return whether the point is within the box. */
    bit contains (vec2 point)
    {
        if (isnan ())
            return false;
        return point.cmpge (min) && point.cmple (max);
    }

    /** Return whether this box fully envelopes b. */
    bit contains (box2 box)
    {
        if (isnan () || box.isnan ())
            return false;
        return box.min.cmpge (min) && box.max.cmple (max);
    }

    /** Get the dimensions of the box. */
    vec2 dim ()
    {
        return max - min;
    }

    /** Alter the dimensions of the box without changing the center. */
    void dim (vec2 value)
    {
        set (center () - value / 2, center () + value / 2);
    }

    /** Ensure that the bounds contain this point. */
    void ensure (vec2 point)
    {
        if (isnan ())
            min = max = point;
        else
        {
            min = min.min (point);
            max = max.max (point);
        }
    }

    /** Ensure that the bounds contain this box. */
    void ensure (box2 box)
    {
        if (isnan ())
            *this = box;
        else if (!box.isnan ())
        {
            min = min.min (box.min);
            max = max.max (box.max);
        }
    }

    /** Return whether this is an uninitialised box. */
    bit isnan ()
    {
        return min.isnan () || max.isnan ();
    }

    /** Return whether this box overlaps the other box. */
    bit overlaps (box2 box)
    {
        if (isnan () || box.isnan ())
            return false;
        return box.max.cmpge (min) && box.min.cmple (max);
    }

    /** Return the nearest point in the box to the vector. */
    vec2 nearestPoint (vec2 vector)
    {
        return avec2 (fmid (min.x, vector.x, max.x), fmid (min.y, vector.y, max.y));
    }

    /** Return an indexed corner of the box.
      * @param index The corner index, from 0 to 3.  Out-of-range values return NAN.
      * @return The corner.
      */
    vec2 corner (uint index)
    {
        switch (index)
        {
            case 0: return min;
            case 1: return avec2 (max.x, min.y);
            case 2: return avec2 (min.x, max.y);
            case 3: return avec2 (max.x, max.y);
            default: return vec2.nan;
        }
    }

    /** Convert to std.string. */
    char [] toString ()
    {
        char [256] buffer;
        int length;

        length = std.c.stdio.sprintf (buffer, "box2 ((%g, %g), (%g, %g))", min.x, min.y, max.x, max.y);
        return buffer [0 .. length].dup;
    }

/** @name Operator overloading.
  *//**@{*/

    /** box2 = (box2 + vec2) */
    box2 opAdd (vec2 value) { return create (min + value, max + value); }

    /** box2 = (box2 += vec2) */
    box2 opAddAssign (vec2 value) { min += value; max += value; return *this; }

    /** box2 = (box2 - vec2) */
    box2 opSub (vec2 value) { return create (min - value, max - value); }

    /** box2 = (box2 -= vec2) */
    box2 opSubAssign (vec2 value) { min -= value; max -= value; return *this; }

/**@}*/
}

/** Create a box with a specific bound. 
  * @relates box2
  */
/+box2 abox2 (vec2 vector)
{
    return box2.create (vector);
}

/** Create a box with specific bounds. 
  * @relates box2
  */
box2 abox2 (vec2 min, vec2 max)
{
    return box2.create (min, max);
}+/

unittest
{
    box2 box;

 //   assert (!box.contains (avec2 (0, 0)));

  //  box.set (avec2 (0, 0), avec2 (1, 1));
//    assert (!box.contains (avec2 (-1, 0)));
    //assert (box.contains (avec2 (0.5, 0)));
    //assert (!box.contains (avec2 (2, 2)));
}

/** A three-dimensional bounding box. */
struct box3
{
    vec3 min; /**< Starting bounds. */
    vec3 max; /**< Ending bounds. */

    /** Create the box with a specific value. */
    static box3 create (vec3 vector)
    {
        return create (vector, vector);
    }

    /** Create a box with specific values. */
    static box3 create (vec3 min, vec3 max)
    {
        box3 result;

        result.set (min, max);
        return result;
    }

    /** Create a box by specifying the center and the length of each side. */
    static box3 fromCenter (vec3 center, vec3 length)
    {
        length = length.abs () / 2;
        return create (center - length, center + length);
    }

    /** Assign specific values. */
    void set (vec3 min, vec3 max)
    {
        this.min = min;
        this.max = max;
    }

    /** Return the center of the box. */
    vec3 center ()
    {
        return (min + max) / 2;
    }

    /** Change the center of the box. */
    void center (vec3 value)
    {
        *this += value - center ();
    }
    
    /** Return whether this point is conservatively within the box. */
    bit concontains (vec3 point)
    {
        if (isnan ())
            return false;
        return point.cmpg (min) && point.cmpl (max);
    }

    /** Return whether this box conservatively envelopes b (greater than rather than greater than or equal to) .*/
    bit concontains (box3 box)
    {
        if (isnan () || box.isnan ())
            return false;
        return box.min.cmpg (min) && box.max.cmpl (max);
    }

    /** Return whether the box conservatively overlaps the other box. */
    bit conoverlaps (box3 box)
    {
        if (isnan () || box.isnan ())
            return false;
        return box.max.cmpg (min) && box.min.cmpl (max);
    }

    /** Return whether the point is within the box. */
    bit contains (vec3 point)
    {
        if (isnan ())
            return false;
        return point.cmpge (min) && point.cmple (max);
    }

    /** Return whether this box fully envelopes b. */
    bit contains (box3 box)
    {
        if (isnan () || box.isnan ())
            return false;
        return box.min.cmpge (min) && box.max.cmple (max);
    }

    /** Get the dimensions of the box. */
    vec3 dim ()
    {
        return max - min;
    }

    /** Alter the dimensions of the box without changing the center. */
    void dim (vec3 value)
    {
        set (center () - value / 2, center () + value / 2);
    }

    /** Ensure that the bounds contain this point. */
    void ensure (vec3 point)
    {
        if (isnan ())
            min = max = point;
        else
        {
            min = min.min (point);
            max = max.max (point);
        }
    }

    /** Ensure that the bounds contain this box. */
    void ensure (box3 box)
    {
        if (isnan ())
            *this = box;
        else if (!box.isnan ())
        {
            min = min.min (box.min);
            max = max.max (box.max);
        }
    }

    /** Return whether this is an uninitialised box. */
    bit isnan ()
    {
        return min.isnan () || max.isnan ();
    }

    /** Return whether this box overlaps the other box. */
    bit overlaps (box3 box)
    {
        if (isnan () || box.isnan ())
            return false;
        return box.max.cmpge (min) && box.min.cmple (max);
    }

    /** Return whether the sphere overlaps or is within the box.
      * @param sphere The sphere to compare against.
      * @return Whether this sphere overlaps the box.
      */
    bit overlaps (sphere3 sphere)
    {
        return sphere.contains (nearestPoint (sphere.center));
    }

    /** Return whether the oriented bounding box overlaps or is within the box.
      * @param obb The oriented bounding box to compare against.
      * @return Whether this oriented bounding box overlaps the box.
      */
    bit overlaps (obb3 obb)
    {
        return aobb3 (*this).overlaps (obb);
    }

    /** Return the nearest point in the box to the vector. */
    vec3 nearestPoint (vec3 vector)
    {
        return avec3 (fmid (min.x, vector.x, max.x), fmid (min.y, vector.y, max.y), fmid (min.z, vector.z, max.z));
    }

    /** Return an indexed corner of the box.
      * @param index The corner index, from 0 to 7.  Out-of-range values return NAN.
      * @return The corner.
      */
    vec3 corner (uint index)
    {
        switch (index)
        {
            case 0: return min;
            case 1: return avec3 (min.x, min.y, max.x);
            case 2: return avec3 (min.x, max.y, min.x);
            case 3: return avec3 (min.x, max.y, max.x);
            case 4: return avec3 (max.x, min.y, min.x);
            case 5: return avec3 (max.x, min.y, max.x);
            case 6: return avec3 (max.x, max.y, min.x);
            case 7: return max;
            default: return vec3.nan;
        }
    }

    /** Return the sphere that encapsulates the box.
      * @return The sphere that encapsulates the box.  If the box is nan, the sphere is nan.
      */
    sphere3 toSphere ()
    {
        vec3 size = this.dim () / 2;
        vec3 center = this.center ();
        float radius = size.magnitude ();

        return asphere3 (center, radius);
    }

    /** Convert to std.string. */
    char [] toString ()
    {
        char [256] buffer;
        int length;

        length = std.c.stdio.sprintf (buffer, "box3 ((%g, %g, %g), (%g, %g, %g))", min.x, min.y, min.z, max.x, max.y, max.z);
        return buffer [0 .. length].dup;
    }

/** @name Operator overloading.
  *//**@{*/

    /** box3 = (box3 + vec3) */
    box3 opAdd (vec3 value) { return create (min + value, max + value); }

    /** box3 = (box3 += vec3) */
    box3 opAddAssign (vec3 value) { min += value; max += value; return *this; }

    /** box3 = (box3 - vec3) */
    box3 opSub (vec3 value) { return create (min - value, max - value); }

    /** box3 = (box3 -= vec3) */
    box3 opSubAssign (vec3 value) { min -= value; max -= value; return *this; }

/**@}*/
}

/** Create a box with a specific bound. 
  * @relates box3
  */
box3 abox3 (vec3 vector)
{
    return box3.create (vector);
}

/** Create a box with specific bounds. 
  * @relates box3
  */
box3 abox3 (vec3 min, vec3 max)
{
    return box3.create (min, max);
}

unittest
{
    box3 box;

    assert (!box.contains (avec3 (0, 0)));

    box.set (avec3 (0, 0, 0), avec3 (1, 1, 1));
    assert (!box.contains (avec3 (-1, 0, 0)));
    assert (box.contains (avec3 (0.5, 0, 0)));
    assert (!box.contains (avec3 (2, 2, 2)));
}

/** An oriented three-dimensional bounding box. */
struct obb3
{
    box3 box; /**< The base box. */
    mat3 axis; /**< Orientation axis.  The OBB pivots around the center of box. */

    /** Create a spoke oriented bounding box.
      * @param centerTop The center of the top of the bounding box.
      * @param axis The orientation axis.
      * @param size The dimensions of the bounding box.
      * @return An oriented bounding box describing these parameters.
      */
    static obb3 createSpoke (vec3 centerTop, mat3 axis, vec3 size)
    {
        vec3 center = centerTop - axis.transpose () * avec3 (0, 0, size.z / 2);
        box3 box;

        box.min = center - size / 2;
        box.max = center + size / 2;

        return aobb3 (box, axis);
    }

    /** Return the center of the box and the pivot for the axis. */
    vec3 center ()
    {
        return box.center ();
    }

    /** Return the dimensions of the box. */
    vec3 dim ()
    {
        return box.dim ();
    }

    /** Return a sphere that encapsulates the box. */
    sphere3 toSphere ()
    {
        return box.toSphere ();
    }

    /** Convert the oriented bounding box into six planes.
      * @param planes The planes to store the box into.  This must have a length of six.
      */

    void toPlanes (plane3[] planes)
    {
        vec3 center;
        vec3 size;

        size = this.dim () / 2;
        center = this.center ();
        
        for (int i = 0; i < 3; i ++)
        {
            float d = center.dot (axis.row (i));

            planes [i * 2] = aplane3 (axis.row (i), d - size.index (i));
            planes [i * 2 + 1] = aplane3 (-axis.row (i), -d - size.index (i));
        }
    }

    /** Convert a local coordinate to world coordinates. */
    vec3 toWorld (vec3 vector)
    {
        vec3 center = box.center ();
        return axis.transpose () * (vector - center) + center;
    }

    /** Convert a world coordinate to local coordinates. */
    vec3 toLocal (vec3 vector)
    {
        vec3 center = box.center ();
        return axis * (vector - center) + center;
    }

    /** Return whether this world vector is inside the box. */
    bit contains (vec3 vector)
    {
        return box.contains (toLocal (vector));
    }

    /** Return an indexed corner of the box in world coordinates. */
    vec3 corner (uint index)
    {
        return toWorld (box.corner (index));
    }

    /** Return whether the bounding boxes overlap one another.
      * This function was stolen from ODE.
      * @param other The other oriented bounding box to test against.
      * @return Whether the bounding boxes overlap one another.
      */

    bit overlaps (obb3 other)
    {
        return overlapsBase (this, other);
    }

    /** Helper function for overlaps that gets around a bug in nested functions. */
    private static bit overlapsBase (obb3* a, obb3 b)
    {
        vec3 p1 = a.box.center ();
        vec3 p2 = b.box.center ();
        mat3 R1 = a.axis;
        mat3 R2 = b.axis;
        vec3 side1 = a.box.dim ();
        vec3 side2 = b.box.dim ();
        vec3 p, pp, normalC;
        float* r = null;
        float A1, A2, A3, B1, B2, B3, R11, R12, R13, R21, R22, R23, R31, R32, R33, Q11, Q12, Q13, Q21, Q22, Q23, Q31, Q32, Q33, s, s2, l;
        int i, invertNormal;

        p = p2 - p1;
        pp = R1 * p;

        A1 = side1.x * 0.5; A2 = side1.y * 0.5; A3 = side1.z * 0.5;
        B1 = side2.x * 0.5; B2 = side2.y * 0.5; B3 = side2.z * 0.5;

        R11 = R1.row (0).dot (R2.row (0)); R12 = R1.row (0).dot (R2.row (1)); R13 = R1.row (0).dot (R2.row (2));
        R21 = R1.row (1).dot (R2.row (0)); R22 = R1.row (1).dot (R2.row (1)); R23 = R1.row (1).dot (R2.row (2));
        R31 = R1.row (2).dot (R2.row (0)); R32 = R1.row (2).dot (R2.row (1)); R33 = R1.row (2).dot (R2.row (2));

        Q11 = std.math.fabs (R11); Q12 = std.math.fabs (R12); Q13 = std.math.fabs (R13);
        Q21 = std.math.fabs (R21); Q22 = std.math.fabs (R22); Q23 = std.math.fabs (R23);
        Q31 = std.math.fabs (R31); Q32 = std.math.fabs (R32); Q33 = std.math.fabs (R33);

        bit test (float expr1, float expr2)
        {
            return std.math.fabs (expr1) - expr2 <= 0;
        }

        if (!test (pp.x, (A1 + B1 * Q11 + B2 * Q12 + B3 * Q13))
        || !test (pp.y, (A2 + B1*Q21 + B2*Q22 + B3*Q23))
        || !test (pp.z, (A3 + B1*Q31 + B2*Q32 + B3*Q33))
        || !test (R2.row (0).dot (p), (A1*Q11 + A2*Q21 + A3*Q31 + B1))
        || !test (R2.row (1).dot (p), (A1*Q12 + A2*Q22 + A3*Q32 + B2))
        || !test (R2.row (2).dot (p), (A1*Q13 + A2*Q23 + A3*Q33 + B3))
        || !test (pp.z*R21-pp.y*R31,(A2*Q31+A3*Q21+B2*Q13+B3*Q12))
        || !test (pp.z*R22-pp.y*R32,(A2*Q32+A3*Q22+B1*Q13+B3*Q11))
        || !test (pp.z*R23-pp.y*R33,(A2*Q33+A3*Q23+B1*Q12+B2*Q11))
        || !test (pp.x*R31-pp.z*R11,(A1*Q31+A3*Q11+B2*Q23+B3*Q22))
        || !test (pp.x*R32-pp.z*R12,(A1*Q32+A3*Q12+B1*Q23+B3*Q21))
        || !test (pp.x*R33-pp.z*R13,(A1*Q33+A3*Q13+B1*Q22+B2*Q21))
        || !test (pp.y*R11-pp.x*R21,(A1*Q21+A2*Q11+B2*Q33+B3*Q32))
        || !test (pp.y*R12-pp.x*R22,(A1*Q22+A2*Q12+B1*Q33+B3*Q31))
        || !test (pp.y*R13-pp.x*R23,(A1*Q23+A2*Q13+B1*Q32+B2*Q31)))
            return false;
        return true;
    }
}

/** Create an oriented bounding box.
  * @relates obb3
  */
obb3 aobb3 (box3 box, mat3 axis)
{
    obb3 result;

    result.box = box;
    result.axis = axis;
    return result;
}

/** Create an oriented bounding box with an identity axis.
  * @relates obb3
  */
obb3 aobb3 (box3 box)
{
    obb3 result;

    result.box = box;
    return result;
}

/** A three-dimensional ray (origin plus direction). */
struct ray3
{
    vec3 origin; /**< Start of the ray. */
    vec3 dir; /**< Unit vector pointing in the direction of the ray. */
    float maximum; /**< Maximum distance for valid strikes or nan for none. */
    float maximumSquared; /**< maximum * maximum or nan for none. */

    /** Assign parameters; dir will be normalized. */
    void set (vec3 origin, vec3 dir)
    {
        this.origin = origin;
        this.dir = dir.normalize ();
        this.maximum = maximum;
        this.maximumSquared = maximum * maximum;
    }

    /** Assign parameters; dir will be normalized. */
    void set (vec3 origin, vec3 dir, float maximum)
    {
        this.origin = origin;
        this.dir = dir.normalize ();
        this.maximum = maximum;
        this.maximumSquared = maximum * maximum;
    }

    /** Assign parameters from a line segment. */
    void setSegment (vec3 a, vec3 b)
    {
        set (a, b - a, a.distance (b));
    }

    /** Return whether a distance is within the range. */
    bit within (float distance)
    {
        if (maximum != maximum)
            return true;
        return distance < maximum;
    }

    /** Attempt to strike box with the ray.  Returns nan if no strike occurred. */
    vec3 strike (box3 box)
    {
        /* Ultimately by Andrew Woo from Graphics Gems. */
        const int RIGHT = 0;
        const int LEFT = 1;
        const int MIDDLE = 2;

        vec3 nan;
        float [3] maxt, candidate, minb, maxb, origin, dir, coord;
        int inside = 1, which;
        vec3 vector;
        int [3] quadrant;
        int c;
  
        minb [0] = box.min.x, minb [1] = box.min.y, minb [2] = box.min.z;
        maxb [0] = box.max.x, maxb [1] = box.max.y, maxb [2] = box.max.z;
        origin [0] = this.origin.x; origin [1] = this.origin.y; origin [2] = this.origin.z;
        dir [0] = this.dir.x, dir [1] = this.dir.y, dir [2] = this.dir.z;
        
        /* Find candidate planes */
        for (c = 0; c < 3; c ++)
        {
            if (origin [c] < minb [c])
            {
                quadrant [c] = LEFT;
                candidate [c] = minb [c];
                inside = 0;
            }
            else if (origin [c] > maxb [c])
            {
                quadrant [c] = RIGHT;
                candidate [c] = maxb [c];
                inside = 0;
            }
            else
                quadrant [c] = MIDDLE;
        }
        
        /* Ray is inside the bounding box */
        if (inside)
            return this.origin;
        
        /* Calculate distances to candidates */
        for (c = 0; c < 3; c ++)
        {
            if (quadrant [c] != MIDDLE && dir [c] != 0)
                maxt [c] = (candidate [c] - origin [c]) / dir [c];
            else
                maxt [c] = -1;
        }
        
        /* Get largest of these distances for final choice */
        which = 0;
        for (c = 1; c < 3; c ++)
        {
            if (maxt [which] < maxt [c])
                which = c;
        }
            
        /* Check that the candidate is really inside */
        if (maxt [which] < 0)
            return nan;
        
        if (!within (maxt [which]))
            return nan;
        
        for (c = 0; c < 3; c ++)
        {
            if (which != c)
            {
                coord [c] = origin [c] + maxt [which] * dir [c];
                if (coord [c] < minb [c] || coord [c] > maxb [c])
                    return nan;
            }
            else
                coord [c] = candidate [c];
        }
        
        return avec3 (coord [0], coord [1], coord [2]);
    }

    /** Attempt to strike plane with the ray.  Returns nan if no strike occurred. */
    vec3 strike (plane3 plane)
    {
        vec3 nan;
        float denom, t;

        denom = plane.normal.dot (dir);
        if (denom == 0) // Ray and plane are parallel. */
            return nan;

        t = -(plane.offset + plane.normal.dot (origin)) / denom;

        // If negative (or close enough) the polygon is behind the ray.
        if (t <= 0.0001 || !within (t))
            return nan;

        return origin + t * dir;
    }

    /** Attempt to strike plane with the ray and return the distance or nan if no strike occurred. */
    float strikeDistance (plane3 plane)
    {
        float denom, t;

        denom = plane.normal.dot (dir);
        if (denom == 0) // Ray and plane are parallel. */
            return float.nan;

        t = -(plane.offset + plane.normal.dot (origin)) / denom;

        // If negative (or close enough) the polygon is behind the ray.
        if (t <= 0.0001 || !within (t))
            return float.nan;

        return t;
    }

    /** Attempt to strike triangle with the ray.  Returns nan if no strike occurred. */
    vec3 strike (vec3 a, vec3 b, vec3 c)
    {
        vec3 [3] polygon;

        polygon [0] = a;
        polygon [1] = b;
        polygon [2] = c;
        return strike (polygon);
    }

    private float v (vec3 [] points, int p, int n) { return n == 0 ? points [p].x : n == 1 ? points [p].y : n == 2 ? points [p].z : 0; }

    /** Attempt to strike convex polygon with the ray.  Returns nan if no strike occurred. */
    vec3 strike (vec3 [] points)
    {
        float u0, v0, u1, v1, u2, v2, alpha, beta;
        float [3] p;
        int inter, i, i1, i2;
        plane3 plane;
        vec3 nan, phit;

        if (points.length < 3)
            return nan;
        plane = aplane3 (points);
        phit = strike (plane);
        if (phit.isnan ())
            return nan;
        i1 = plane.i1, i2 = plane.i2;

        p [] = phit.array ();
        u0 = p [i1] - v (points, 0, i1);
        v0 = p [i2] - v (points, 0, i2);
        inter = 0;
        i = 2;

        do
        {
            u1 = v (points, i - 1, i1) - v (points, 0, i1);
            v1 = v (points, i - 1, i2) - v (points, 0, i2);
            u2 = v (points, i, i1) - v (points, 0, i1);
            v2 = v (points, i, i2) - v (points, 0, i2);
            
            if (u1 == 0)
            {
                beta = u0 / u2;
                if (beta >= 0 && beta <= 1)
                {
                    alpha = (v0 - beta * v2) / v1;
                    inter = (alpha >= 0 && alpha + beta <= 1);
                }
            }
            else
            {
                beta = (v0 * u1 - u0 * v1) / (v2 * u1 - u2 * v1);
                if (beta >= 0 && beta <= 1)
                {
                    alpha = (u0 - beta * u2) / u1;
                    inter = (alpha >= 0 && alpha + beta <= 1);
                }
            }
        }
        while (!inter && ++ i < points.length);

        if (inter)
            return phit;
        return nan;
    }
}

/** Create a three-dimensional ray; dir will be normalized.
  * @relates ray3
  */
ray3 aray3 (vec3 origin, vec3 dir)
{
    ray3 result;

    result.set (origin, dir);
    return result;
}

/** Create a three-dimensional ray; dir will be normalized.
  * @relates ray3
  */
ray3 aray3 (vec3 origin, vec3 dir, float maximum)
{
    ray3 result;

    result.set (origin, dir, maximum);
    return result;
}

/** Create a three-dimensional array from a line segment. 
  * @relates ray3
  */
ray3 aray3segment (vec3 a, vec3 b)
{
    ray3 result;

    result.setSegment (a, b);
    return result;
}

/** A three-dimensional plane. */
struct plane3
{
    vec3 normal; /**< Plane normal. */
    float offset; /**< Distance of the plane from origin along the normal. */
    int i1; /**< For collision with rays. */
    int i2; /**< For collision with rays. */

    /** Extrapolate plane from a triangle. */
    void set (vec3 pa, vec3 pb, vec3 pc)
    {
        float a, b, c, d, e, f, g, h, i;

        a = pb.x - pa.x; b = pb.y - pa.y; c = pb.z - pa.z;
        d = pc.x - pb.x; e = pc.y - pb.y; f = pc.z - pb.z;
        normal = avec3 (b * f - c * e, c * d - a * f, a * e - d * b);

        offset = -(g * pa.x + h * pa.y + i * pa.z);

        set (normal, offset);
    }
  
    /** Extrapolate plane from a polygon. */
    void set (vec3 [] points)
    {
        assert (points.length >= 3);
        set (points [0], points [1], points [points.length - 1]);
    }

    /** Assign parameters; normal does not have to be a unit vector. */
    void set (vec3 normal, float offset)
    {
        float a, b, c, g, h, i;

        this.normal = normal.normalize ();
        this.offset = offset;

        g = normal.x; h = normal.y; i = normal.z;
        a = (g < 0 ? -g : g);
        b = (h < 0 ? -h : h);
        c = (i < 0 ? -i : i);

        if (a < b)
        {
            i1 = 0;
            i2 = (b < c) ? 1 : 2;
        }
        else
        {
            i1 = 1;
            i2 = (a < c) ? 0 : 2;
        }
    }

    /** Return the signed distance from the vector to the plane. */
    float signedDistance (vec3 vector)
    {
        return normal.dot (vector) - offset;
    }

    /** Return the distance from the vector to the plane. */
    float distance (vec3 vector)
    {
        return normal.dot (vector) - offset;
    }

    /** Return whether this point is in front of the plane. */
    bit inFront (vec3 vector)
    {
        return signedDistance (vector) >= 0;
    }

    /** Return a string representation of the plane, in the form "plane3 ((normal), distance)". */
    char[] toString ()
    {
        return fmt ("plane ((%g, %g, %g), %g)", normal.x, normal.y, normal.z, offset);
    }

    /** Get the intersection between a line segment and the plane.
      * @param a The start of the line segment.
      * @param b The end of the line segment.
      * @param point The point of intersection is stored here if the plane intersects the line segment.
      * @return Returns the intersection blend - from 0 (meaning a) to 1 (meaning b) - or float.nan if there is no intersection.
      * It returns float.infinity if the line is on the plane.
      */

    float intersectLineSegment (vec3 a, vec3 b, out vec3 point)
    {
        float u, v, w;

        v = normal.x * (a.x - b.x) + normal.y * (a.y - b.y) + normal.z * (a.z - b.z);
        if (v == 0)
            return float.infinity;
        w = normal.x * a.x + normal.y * a.y + normal.z * a.z - offset;
        u = w / v;

        if (u < 0 || u > 1)
            return float.nan;

        point = a * (1 - u) + b * u;
        return u;
    }

    /** Get the intersection point between a line segment and the plane.
      * @param a The start of the line segment.
      * @param b The end of the line segment.
      * @return Returns the intersection blend - from 0 (meaning a) to 1 (meaning b) - or float.nan if there is no intersection.
      * It returns float.infinity if the line is on the plane.
      */

    float intersectLineSegment (vec3 a, vec3 b)
    {
        float u, v, w;

        v = normal.x * (a.x - b.x) + normal.y * (a.y - b.y) + normal.z * (a.z - b.z);
        if (v == 0)
            return float.infinity;
        w = normal.x * a.x + normal.y * a.y + normal.z * a.z - offset;
        u = w / v;

        if (u < 0 || u > 1)
            return float.nan;
        return u;
    }

    /** Get the relation between the plane and a sphere.
      * @param sphere The sphere to test.  If it is NAN, the return value is undefined.
      * @return +1 if the sphere is fully in front of the plane, 0 if the sphere is crossing the plane, or -1 if the sphere is fully behind the plane.
      */

    int classify (sphere3 sphere)
    {
        float d = signedDistance (sphere.center);

        if (d > sphere.radius)
            return 1;
        if (d < -sphere.radius)
            return -1;
        return 0;
    }

    /** Get the relation between the plane and a vector.
      * @param vector The vector to test.
      * @return +1 if the vector is fully in front of the plane, 0 if the vector is on the plane, or -1 if the vector is fully behind the plane.
      */

    int classify (vec3 vector)
    {
        float d = signedDistance (vector);

        if (d > 0)
            return 1;
        if (d < 0)
            return -1;
        return 0;
    }

    /** Get the relation between the plane and an axis-aligned bounding box.
      * @param box The axis-aligned bounding box to test.
      * @return +1 if the box is fully in front of the plane, 0 if the box is crossing the plane, or -1 if the box is fully behind the plane.
      */

    int classify (box3 box)
    {
        return classifyBase (this, box);
    }

    /** Helper function for classify that gets around a bug in nested functions. */
    static private int classifyBase (plane3* plane, box3 box)
    {
        vec3 min = box.min;
        vec3 max = box.max;
        vec3 center = box.center ();
        int result;

        int test (vec3 vector)
        {
            return plane.classify (vector);
        }

        int opAdd (vec3 vector)
        {
            if (test (vector) != result)
                result = 0;
            return result;
        }

        result = test (min);
        if (!opAdd (max)
         || !opAdd (avec3 (min.x, min.y, max.z))
         || !opAdd (avec3 (min.x, max.y, min.z))
         || !opAdd (avec3 (min.x, max.y, max.z))
         || !opAdd (avec3 (max.x, min.y, min.z))
         || !opAdd (avec3 (max.x, min.y, max.z))
         || !opAdd (avec3 (max.x, max.y, min.z)))
            return result;
        return result;
    }

    /** Get the relation between the plane and an oriented bounding box.
      * @param box The oriented bounding box to test.
      * @return +1 if the box is fully in front of the plane, 0 if the box is crossing the plane, or -1 if the box is fully behind the plane.
      */

    int classify (obb3 box)
    {
        //classify (box.axis.transpose () * (vector - center) + center);
        return classifyBase (this, box);
    }

    /** Helper function for classify that gets around a bug in nested functions. */
    static private int classifyBase (plane3* plane, obb3 box)
    {
        vec3 min = box.box.min;
        vec3 max = box.box.max;
        vec3 center = box.center ();
        mat3 axis = box.axis.transpose ();
        int result;

        int test (vec3 vector)
        {
            return plane.classify (axis * (vector - center) + center);
        }

        int opAdd (vec3 vector)
        {
            if (test (vector) != result)
                result = 0;
            return result;
        }

        result = test (min);
        if (!opAdd (max)
         || !opAdd (avec3 (min.x, min.y, max.z))
         || !opAdd (avec3 (min.x, max.y, min.z))
         || !opAdd (avec3 (min.x, max.y, max.z))
         || !opAdd (avec3 (max.x, min.y, min.z))
         || !opAdd (avec3 (max.x, min.y, max.z))
         || !opAdd (avec3 (max.x, max.y, min.z)))
            return result;
        return result;
    }
}

/** Create a plane from a triangle. 
  * @relates plane3
  */
plane3 aplane3 (vec3 a, vec3 b, vec3 c)
{
    plane3 result;

    result.set (a, b, c);
    return result;
}

/** Create a plane from a polygon. 
  * @relates plane3
  */
plane3 aplane3 (vec3 [] points)
{
    plane3 result;

    result.set (points);
    return result;
}

/** Create a plane from specific parameters; normal does not have to be a unit vector.
  * @relates plane3
  */
plane3 aplane3 (vec3 normal, float offset)
{
    plane3 result;

    result.set (normal, offset);
    return result;
}

/** A three-dimensional sphere. */
struct sphere3
{
    vec3 center; /**< Center of the sphere. */
    float radius; /**< Sphere radius. */

    /** Return whether the point is within the sphere. */
    bit contains (vec3 point)
    {
        return center.squaredDistance (point) <= radius * radius;
    }

    /** Return whether the sphere is within the sphere. */
    bit contains (sphere3 sphere)
    {
        return center.squaredDistance (sphere.center) + sphere.radius * sphere.radius <= radius * radius;
    }

    /** Resize the sphere to contain the point. */
    void ensure (vec3 point)
    {
        float distance = center.squaredDistance (point);

        if (distance <= radius * radius)
            return;
        radius = std.math.sqrt (distance);
    }

    /** Resize the sphere to contain the other sphere. */
    void ensure (sphere3 sphere)
    {
        float distance = center.squaredDistance (sphere.center) + sphere.radius * sphere.radius;

        if (distance <= radius * radius)
            return;
        radius = std.math.sqrt (distance);
    }

    /** Return whether this has not been initialized. */
    bit isnan ()
    {
        return center.isnan () || radius != radius;
    }

    /** Assign parameters. */
    void set (vec3 center, float radius)
    {
        this.center = center;
        this.radius = radius;
    }

    /** Return whether this and that sphere overlap one another.
      * @param other The other sphere to test for overlap.
      * @return Whether the spheres are within one another's bounds.
      */
    bit overlaps (sphere3 other)
    {
        float distance = center.distance (other.center);

        return distance <= radius + other.radius;
    }
}

/** Create a sphere with the specified parameters.
  * @relates sphere3
  */
sphere3 asphere3 (vec3 center, float radius)
{
    sphere3 result;

    result.set (center, radius);
    return result;
}

/** A quaternion object.  The typical usage is to create an instance using
  * quat3.rotationDeg or quat3.rotationRad.  You can then convert it to a matrix
  * using .tomat4 (matrices are faster at multiplication), multiply them together,
  * multiply them against vectors, or interpolate them using #lerpShort, #lerpLong,
  * #lerpCW, #lerpCCW, #lerpLinear, or #lerp.
  */

struct quat3
{
    float w = 1; /**< W factor. */
    float x = 0; /**< X factor. */
    float y = 0; /**< Y factor. */
    float z = 0; /**< Z factor. */

    /** Parameter for #lerp. */
    enum Lerp
    {
        Short, /**< Use the shortest path. */
        Long, /**< Use the longest path. */
        CW, /**< Rotate clockwise when viewed from above. */
        CCW, /**< Rotate counter-clockwise when viewed from above. */
        Linear, /**< Merely linearly interpolate the values. */
    }

    /** Return a quaternion that rotates a number of degrees around each axis. */
    static quat3 rotationDeg (float x, float y, float z)
    {
        vec3 v = avec3 (x, y, z);
        return rotationRad (v * std.math.PI / 180);
    }

    /** Return a quaternion that rotates a number of degrees around each axis. */
    static quat3 rotationDeg (vec3 amount)
    {
        return rotationDeg (amount.x, amount.y, amount.z);
    }

    /** Return a quaternion that rotates around an axis by a number of degrees. */
    static quat3 rotationDeg (float amount, float x, float y, float z)
    {
        return rotationRad (amount * std.math.PI / 180, avec3 (x, y, z));
    }

    /** Return a quaternion that rotates around an axis by a number of degrees. */
    static quat3 rotationDeg (float amount, vec3 axis)
    {
        return rotationRad (amount * std.math.PI / 180, axis);
    }

    /** Return a quaternion that rotates a number of radians around each axis. */
    static quat3 rotationRad (float x, float y, float z)
    {
        float sx, sy, sz;
        float cx, cy, cz;
        quat3 quat;

        sx = std.math.sin (x / 2);
        sy = std.math.sin (y / 2);
        sz = std.math.sin (z / 2);
        cx = std.math.cos (x / 2);
        cy = std.math.cos (y / 2);
        cz = std.math.cos (z / 2);
        
        quat.w = (cx * cy * cz) + (sx * sy * sz);
        quat.x = (sx * cy * cz) - (cx * sy * sz);
        quat.y = (cx * sy * cz) + (sx * cy * sz);
        quat.z = (cx * cy * sz) - (sx * sy * cz);
        return quat;
    }

    /** Return a quaternion that rotates a number of radians around each axis. */
    static quat3 rotationRad (vec3 amount)
    {
        return rotationRad (amount.x, amount.y, amount.z);
    }

    /** Return a quaternion that rotates around an axis by a number of radians. */
    static quat3 rotationRad (float amount, float x, float y, float z)
    {
        return rotationRad (amount, avec3 (x, y, z));
    }

    /** Return a quaternion that rotates around an axis by a number of radians. */
    static quat3 rotationRad (float amount, vec3 axis)
    {
        quat3 quat;
        float s;

        axis = axis.normalize ();
        s = std.math.sin (amount / 2);
        quat.w = std.math.cos (amount / 2);
        quat.x = s * axis.x;
        quat.y = s * axis.y;
        quat.z = s * axis.z;
        return quat;
    }

    /** Return the quaternion component from a 4x4 matrix. */
    static quat3 frommat4 (mat4 matrix)
    {
        return matrix.toquat3 ();
    }

    /** Multiply quaternions, so that the result is like applying this quaternion and then the next one. */
    quat3 opMul (quat3 b)
    {
        quat3 quat;

        quat.w = (w * b.w) - (x * b.x) - (y * b.y) - (z * b.z);
        quat.x = (x * b.x) + (x * b.w) + (y * b.z) - (z * b.y);
        quat.y = (w * b.y) + (y * b.w) + (z * b.x) - (x * b.z);
        quat.z = (w * b.z) + (z * b.w) + (x * b.y) - (y * b.x);
        return quat;
    }

    /** Rotate vector by this quaternion.  This is a slow operation - convert this to a matrix
      * (using #tomat3 or #tomat4) and apply that if you are going to be rotating many vectors.
      */
    vec3 opMul (vec3 vector)
    {
        quat3 t, i, v;

        v = aquat3 (0, vector.x, vector.y, vector.z);
        i = this.inverse ();
        t = i * v;
        v = t * *this;

        return avec3 (v.x, v.y, v.z);
    }

    /** Convert the quaternion to a 4x4 matrix. */
    mat4 tomat4 ()
    {
        mat4 matrix;
        float ww, xx, yy, zz, wx, wy, wz, xy, xz, yz;

        /*  | ww + xx - yy - zz       2xy + 2wz             2xz - 2wy     |
         *  |     2xy - 2wz       ww - xx + yy - zz         2yz - 2wx     |
         *  |     2xz + 2wy           2yz - 2wx         ww + xx - yy - zz |
         */
        
        ww = w * w;
        xx = x * x;
        yy = y * y;
        zz = z * z;
        wx = w * x * 2;
        wy = w * y * 2;
        wz = w * z * 2;
        xy = x * y * 2;
        xz = x * z * 2;
        yz = y * z * 2;

        matrix.row (0, ww + xx - yy - zz, xy - wz, xz + wy, 0);
        matrix.row (1, xy + wz, ww - xx + yy - zz, yz - wx, 0);
        matrix.row (2, xz - wy, yz + wx, ww - xx - yy + zz, 0);
        matrix.row (3, 0, 0, 0, 1);
        
        return matrix;
    }

    /** Convert the quaternion to a 3x3 matrix. */
    mat3 tomat3 ()
    {
        mat3 matrix;
        float ww, xx, yy, zz, wx, wy, wz, xy, xz, yz;

        /*  | ww + xx - yy - zz       2xy + 2wz             2xz - 2wy     |
         *  |     2xy - 2wz       ww - xx + yy - zz         2yz - 2wx     |
         *  |     2xz + 2wy           2yz - 2wx         ww + xx - yy - zz |
         */
        
        ww = w * w;
        xx = x * x;
        yy = y * y;
        zz = z * z;
        wx = w * x * 2;
        wy = w * y * 2;
        wz = w * z * 2;
        xy = x * y * 2;
        xz = x * z * 2;
        yz = y * z * 2;

        matrix.row (0, ww + xx - yy - zz, xy - wz, xz + wy);
        matrix.row (1, xy + wz, ww - xx + yy - zz, yz - wx);
        matrix.row (2, xz - wy, yz + wx, ww - xx - yy + zz);
        
        return matrix;
    }

    /** Return the inverse of this quaternion, containing the opposite rotation. */
    quat3 inverse ()
    {
        quat3 conjugate;
        float normal;

        conjugate = this.conjugate ();
        normal = this.normal ();
        
        return aquat3 (conjugate.w / normal, conjugate.x / normal, conjugate.y / normal, conjugate.z / normal);
    }

    /** Interpolate between two quaternions using the shortest path between them. */
    static quat3 lerpShort (quat3 a, quat3 b, float t)
    {
        return lerp (a, b, t, Lerp.Short);
    }

    /** Interpolate between two quaternions using the longest path between them. */
    static quat3 lerpLong (quat3 a, quat3 b, float t)
    {
        return lerp (a, b, t, Lerp.Long);
    }

    /** Interpolate between two quaternions by rotating clockwise when viewed from above. */
    static quat3 lerpCW (quat3 a, quat3 b, float t)
    {
        return lerp (a, b, t, Lerp.CW);
    }

    /** Interpolate between two quaternions by rotating counter-clockwise when viewed from above. */
    static quat3 lerpCCW (quat3 a, quat3 b, float t)
    {
        return lerp (a, b, t, Lerp.CCW);
    }

    /** Interpolate between two quaternions linearly. */
    static quat3 lerpLinear (quat3 a, quat3 b, float t)
    {
        return lerp (a, b, t, Lerp.Linear);
    }

    /** Interpolate between two quaternions. */
    static quat3 lerp (quat3 a, quat3 b, float t, Lerp how)
    {
        float angle, cosAngle, scaleFrom, scaleTo, sinAngle;
        quat3 to, quat;

        cosAngle = (a.x * b.x) + (a.y * b.y) + (a.z * b.z) + (a.w * b.w);
        
        if ((how == Lerp.Short && cosAngle < 0)
         || (how == Lerp.Long && cosAngle > 0)
         || (how == Lerp.CW && a.w > b.w)
         || (how == Lerp.CCW && a.w < b.w))
        {
            cosAngle = -cosAngle;
            to.w = -b.w;
            to.x = -b.x;
            to.y = -b.y;
            to.z = -b.z;
        }
        else
            to = b;
        
        if ((1 - std.math.fabs (cosAngle)) > 0.0001)
        {
            /* Spherical linear interpolation */
            angle = std.math.acos (cosAngle);
            sinAngle = std.math.sin (angle);
            scaleFrom = std.math.sin ((1 - t) * angle) / sinAngle;
            scaleTo = std.math.sin (t * angle) / sinAngle;
        }
        else
        {
            scaleFrom = 1 - t;
            scaleTo = t;
        }
        
        quat.w = (scaleFrom * a.w) + (scaleTo * to.w);
        quat.x = (scaleFrom * a.x) + (scaleTo * to.x);
        quat.y = (scaleFrom * a.y) + (scaleTo * to.y);
        quat.z = (scaleFrom * a.z) + (scaleTo * to.z);
        return quat;
    }

    /** Return the normal of the quaternion. */
    float normal ()
    {
        return w * w + x * x + y * y + z * z;
    }

    /** Return the conjugate of the quaternion (w, -x, -y, -z). */
    quat3 conjugate ()
    {
        return aquat3 (w, -x, -y, -z);
    }
}

/** Assign parameters. */
quat3 aquat3 (float w, float x, float y, float z)
{
    quat3 result;

    result.w = w;
    result.x = x;
    result.y = y;
    result.z = z;
    return result;
}
